import logging
from datetime import datetime, timedelta
from sandbox import sdk2
from sandbox.common import rest
from sandbox.common.types import resource
from sandbox.common.errors import TaskFailure
from sandbox.projects.common import solomon

DBS = ['rus', 'tur']

CHRONO_TYPE = 'ADVQ_NORMAL_CHRONO_DATABASE_CHUNK'
PHITS_TYPE = 'ADVQ_NORMAL_PHITS_DATABASE_CHUNK'

logger = logging.getLogger(__name__)

def day_in_previous_month(dt):
    return dt.replace(day=1) - timedelta(days=1)

def generator_months(months):
    date = datetime.today()
    date = day_in_previous_month(date)
    for _ in range(months):
        date = day_in_previous_month(date)
        yield date.strftime('%Y%m')

def generator_weeks(weeks):
    date = datetime.today()
    date -= timedelta(date.weekday())       # set day to nearest Monday, see ADVQ-2252
    date -= timedelta(days=7)
    for _ in range(weeks):
        date -= timedelta(days=7)
        if date.strftime('%V') == '53':     # because 53rd week is actually considered as 1st week of the next year
            continue
        yield date.strftime('%Y%V')

def generator_days(days):
    date = datetime.today()
    date -= timedelta(days=3)
    for _ in range(days):
        date -= timedelta(days=1)
        yield date.strftime("%Y%m%d")

class AdvqCheckResources(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        month_window = sdk2.parameters.Integer('Months to check chrono', default=26)
        week_window = sdk2.parameters.Integer('Weeks to check chrono', default=70)
        day_window = sdk2.parameters.Integer('Days to check phits', default=30)

        solomon_token_secret = sdk2.parameters.YavSecret('Yav secret with Solomon token', required=True)

    def on_execute(self):
        self.client = rest.Client()

        err = 0

        for db in DBS:
            # Chrono
            for date in generator_months(self.Parameters.month_window):
                try:
                    self._check_resources(db, CHRONO_TYPE, date, 'month')
                except:
                    err = 1
            for date in generator_weeks(self.Parameters.week_window):
                try:
                    self._check_resources(db, CHRONO_TYPE, date, 'week')
                except:
                    err = 1

            # Phits
            for date in generator_days(self.Parameters.day_window):
                try:
                    self._check_resources(db, PHITS_TYPE, date)
                except:
                    err = 1

        logger.info('Check finished, push result to solomon')
        solomon.push_to_solomon_v2(
            self.Parameters.solomon_token_secret.data()['solomon'],
            {
                'project': 'advq',
                'cluster': 'db',
                'service': 'advq_check_resources',
            },
            [
                {
                    'labels': {
                        'sensor': 'err',
                    },
                    'ts': int(datetime.now().strftime('%s')),
                    'value': err,
                }
            ],
        )
        logger.info('Uploaded successfully')

        if err:
            raise TaskFailure('Check failed: some resources not found, check common.log')

    def _check_resources(self, db, resourse_type, epodate, chrono_type=None):
        logger.info('Start checking: {db}, {type}, {epodate}'.format(db=db, type=resourse_type, epodate=epodate) + (', ' + chrono_type if chrono_type is not None else ''))

        attrs = {
            'advq_db': db,
            'advq_epodate': epodate,
        }
        if chrono_type is not None:
            attrs['advq_chrono_type'] = chrono_type
        first_resource = self.client.resource.read(
            type=resourse_type,
            attrs=attrs,
            state='READY',
            limit=1,
        )['items']

        if not len(first_resource):
            logger.error('First resource not found')
            raise

        total_chunks = int(first_resource[0]['attributes']['advq_total_chunks'])

        for chunk in range(1, total_chunks + 1):
            logger.info('Checking: chunk {chunk}'.format(chunk=chunk))

            attrs = {
                'advq_db': db,
                'advq_epodate': epodate,
                'advq_chunk': chunk,
            }
            if chrono_type is not None:
                attrs['advq_chrono_type'] = chrono_type
            resource = self.client.resource.read(
                type=resourse_type,
                attrs=attrs,
                state='READY',
                limit=1,
            )['items']

            if not len(resource):
                logger.error('Chunk not found')
                raise

        logger.info('Check finished successfully')
