# -*- coding: utf-8 -*-
import json
import os
import os.path
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.advq.common.configs import AdvqPhitsConfig, AdvqForecastConfig
from sandbox.projects.advq.common.parameters import convert_ttl
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.platform_api import platform_api
from sandbox.sdk2.resource import ResourceData

CONFIG_DIR = 'data'
CONFIG_FILENAME = 'layout.conf'


class AdvqDeployConfigToHosts(nanny.ReleaseToNannyTask2, platform_api.DeployToPlatformTask, sdk2.Task):
    """
    Вспомогательная таска, которая берёт на входе ресурс с конфигом, пересоздаёт его и релизит в Няню или Платформу,
    в зависимости от аргументов.

    Тип ресурса определяет, куда происходит деплой, и тип остаётся неизменным.

    Входной конфиг может лежать в data/layout.conf или в data{nnnn}/layout.conf, таск работает с обоими случаями.
    Выходной всегда лежит в data/layout.conf.
    """
    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 256
        ram = 256

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        input_resource = sdk2.parameters.Resource("Resource with config", required=True)
        deploy_to_nanny = sdk2.parameters.Bool("Deploy to Nanny", default=False, required=True)
        deploy_to_platform = sdk2.parameters.Bool("Deploy to Platform", default=False, required=True)
        ttl = sdk2.parameters.Integer("TTL for released config resource (0 for inf)", default=30)

        with sdk2.parameters.Output:
            result_config_resource_id = sdk2.parameters.Integer("Generated config Resource ID", required=True)

    @platform_api.DeployToPlatformTask.oauth_token.getter
    def oauth_token(self):
        return sdk2.Vault.data(self.owner, "platform-oauth-token")

    def get_stable_environments(self):
        return self.Parameters.input_resource.platform_stable_environments

    def get_testing_environments(self):
        return self.Parameters.input_resource.platform_testing_environments

    def on_execute(self):
        assert isinstance(self.Parameters.input_resource, AdvqPhitsConfig) or isinstance(self.Parameters.input_resource, AdvqForecastConfig)
        original_res_data = ResourceData(self.Parameters.input_resource)

        new_config = self.Parameters.input_resource.__class__(
            self,
            self.Parameters.input_resource.description,
            CONFIG_DIR,
            **dict(self.Parameters.input_resource)
        )
        # Конфиг расположен в data{NNNN)/layout.conf, path указывает на data{NNNN}
        with open(str(original_res_data.path.joinpath(CONFIG_FILENAME)), 'rb') as old_config_inp:
            config_data = json.load(old_config_inp)

        for field in ('bases', 'layout', 'metas', 'snapshots'):
            if field not in config_data:
                raise TaskFailure("Incorrect config: field {!r} is missing in config #{}".format(
                    field, self.Parameters.input_resource.id,
                ))

        # Устанавливаем индентификатор нового ресурса, т.к. он будет релизнут с нужным TTL.
        config_data['version'] = int(new_config)

        os.mkdir(CONFIG_DIR)
        with open(os.path.join(CONFIG_DIR, CONFIG_FILENAME), 'wb') as output_file:
            json.dump(config_data, output_file, sort_keys=True)
        new_res_data = ResourceData(new_config)
        new_res_data.ready()

        self.Parameters.result_config_resource_id = new_config.id

    def on_release(self, additional_parameters):
        if self.Parameters.deploy_to_nanny:
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        if self.Parameters.deploy_to_platform:
            platform_api.DeployToPlatformTask.on_release(self, additional_parameters)
        # TTL
        self.mark_released_resources(additional_parameters['release_status'], ttl=convert_ttl(self.Parameters.ttl))
