# -*- coding: utf-8 -*-
"""
Генерируем hitlog-таблицы на YT из chhit-log.

Таск автоопределяет то, какие данные доступны для генерации.  Через вызовы утилит ADVQ
он создаёт таблицы на YT, не производя и не релизя никаких ресурсов.

"""
import json
import logging
import tempfile

import sandbox.common.types.client as ctc
import sandbox.sdk2.helpers
from sandbox import sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import HITLOG_FROM_CHHIT_GENERATOR
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.common.yt_utils import get_yt_yql_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

SEMAPHORE_NAME_HITLOG = 'advq_hitlog_table_from_chhit'


class AdvqGenHitlogFromChhitTables(sdk2.Task):
    """
    Build ADVQ hitlog tables from chhit.
    """

    class Requirements(sdk2.Task.Requirements):
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_HITLOG,
                                   capacity=1)
            ]
        )
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        cores = 1
        ram = 256
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String("YT server name", required=True)
        yt_token_vault_user = sdk2.parameters.String("YT key vault owner")
        yt_token_vault_name = sdk2.parameters.String("YT key vault name")
        yql_token_vault_user = sdk2.parameters.String("YQL key vault owner")
        yql_token_vault_name = sdk2.parameters.String("YQL key vault name")
        advq_phits_binaries = sdk2.parameters.LastReleasedResource("ADVQ phits from chhit utils resource",
                                                                   required=True)
        advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")
        custom_hitlogs_path = sdk2.parameters.String("Custom path for generated hitlogs")
        limit = sdk2.parameters.Integer("Limit of tables to generate at once", default=4)
        generate_depth_days = sdk2.parameters.Integer("If hitlogs dir is empty generate N last days", default=10,
                                                      required=True)
        from_hourly = sdk2.parameters.Bool("Generate from 1h chhit logs", default=False)

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_phits_binaries, [YT_MINIMAL_REV])

        advq_phits_binaries_data = ResourceData(self.Parameters.advq_phits_binaries)

        env = get_yt_yql_env_from_parameters(self.Parameters)

        if self.Parameters.advq_yt_prefix:
            config_file = tempfile.NamedTemporaryFile()
            import yaml
            yaml.safe_dump({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            }, config_file.file)
            config_file.flush()
            env['ADVQ_CONFIG_FILE'] = config_file.name

        hitlog_generator_binary_path = str(advq_phits_binaries_data.path.joinpath(HITLOG_FROM_CHHIT_GENERATOR))
        with sandbox.sdk2.helpers.ProcessLog(self,
                                             logger=logging.getLogger("advq-phits-from-chhit-list-sources")) as pl:
            command = [hitlog_generator_binary_path, 'list-gen', '--list-gen-days-depth',
                       str(self.Parameters.generate_depth_days)]
            if self.Parameters.custom_hitlogs_path:
                command = command + ["--hitlogs-path", self.Parameters.custom_hitlogs_path]
            if self.Parameters.from_hourly:
                command += ['--from-hourly']
            list_json_str = sp.check_output(
                command,
                timeout=600,
                stderr=pl.stdout,
                env=env
            )
            to_generate = json.loads(list_json_str)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-from-chhit-generate")) as pl:
            for to_generate_args in to_generate[:self.Parameters.limit]:
                command = [hitlog_generator_binary_path, 'gen'] + [to_generate_args]
                if self.Parameters.custom_hitlogs_path:
                    command = command + ["--output-path", self.Parameters.custom_hitlogs_path]
                if self.Parameters.from_hourly:
                    command += ['--from-hourly']
                sp.check_call(
                    command,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env=env
                )
