# -*- coding: utf-8 -*-
"""
Генерируем hitlog-таблицы на YT из RotateHitLog.

Таск автоопределяет то, какие данные доступны для генерации.  Через вызовы утилит ADVQ
он создаёт таблицы на YT, не производя и не релизя никаких ресурсов.

При запуске утилит берётся семафор с именем advq_hitlog_table с счётчиком 1.
"""
import json
import logging
import tempfile

from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.projects.advq.artifacts import HITLOG_GENERATOR
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.common.types.task import Semaphores
import sandbox.common.types.client as ctc
from sandbox.sdk2.resource import ResourceData


SEMAPHORE_NAME_HITLOG = 'advq_hitlog_table'


class AdvqGenHitlogTables(sdk2.Task):
    """
    Build ADVQ hitlog tables.
    """
    class Requirements(sdk2.Task.Requirements):
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_HITLOG,
                                   capacity=1)
            ]
        )
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        cores = 1
        ram = 256
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String("YT server name", required=True)
        yt_token_vault_user = sdk2.parameters.String("YT key vault owner")
        yt_token_vault_name = sdk2.parameters.String("YT key vault name")
        advq_phits_binaries = sdk2.parameters.LastReleasedResource("ADVQ phits utils resource", required=True)
        advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")
        legacy_output = sdk2.parameters.Bool("Legacy, mrkit-compatible output", default=True)
        limit = sdk2.parameters.Integer("Limit of tables to generate at once", default=4)

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_phits_binaries, [YT_MINIMAL_REV])

        advq_phits_binaries_data = ResourceData(self.Parameters.advq_phits_binaries)

        env = get_yt_env_from_parameters(self.Parameters)

        if self.Parameters.advq_yt_prefix:
            config_file = tempfile.NamedTemporaryFile()
            import yaml
            yaml.safe_dump({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            }, config_file.file)
            config_file.flush()
            env['ADVQ_CONFIG_FILE'] = config_file.name

        hitlog_generator_binary_path = str(advq_phits_binaries_data.path.joinpath(HITLOG_GENERATOR))
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-list-sources")) as pl:
            list_json_str = sp.check_output(
                [hitlog_generator_binary_path, 'list-gen'],
                timeout=600,
                stderr=pl.stdout,
                env=env
            )
            to_generate = json.loads(list_json_str)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-generate")) as pl:
            if self.Parameters.legacy_output:
                legacy_output_args = ['--legacy-output']
            else:
                legacy_output_args = []
            for to_generate_args in to_generate[:self.Parameters.limit]:
                sp.check_call(
                    [hitlog_generator_binary_path, 'gen'
                     ] + legacy_output_args + [to_generate_args],
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env=env
                )
