# -*- coding: utf-8 -*-
"""
Генерация phits-индекса из отдельно взятой таблицы.  Создаются несколько чанков.

Вызывается таском AdvqGenPhitsIndices, который, в свою очередь, вызывается из шедулера.
"""
import logging

from sandbox import sdk2
import sandbox.sdk2.helpers
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import ADVQ_PHITS_DB_GENERATOR
from sandbox.projects.advq.common.yt_utils import setup_yt_from_parameters, get_yt_env_from_parameters
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

from sandbox.projects.advq.common import get_phits_resource_class
from sandbox.projects.advq.common.parameters import PhitsParameters, convert_ttl

SEMAPHORE_GENERATION_NAME_TEMPLATE = 'advq_phits_db_generation_single_{type}'


class AdvqGenPhitsIndex(sdk2.Task):
    """
    Гененрируем чанки для одной таблицы.
    """
    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet")
        )
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        disk_space = 20 * 1024
        ram = 8 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        advq_db = sdk2.parameters.String("Database name", required=True)
        advq_date = sdk2.parameters.String("Date", required=True)
        advq_epoch = sdk2.parameters.Integer("epoch", required=True)
        advq_epodate = sdk2.parameters.String("Date and, perhaps, epoch, formatted as string", required=True)
        advq_is_delta = sdk2.parameters.Bool("Is delta", required=True)
        input_table = sdk2.parameters.String("YT input table path", required=True)
        phits_tmp_prefix = sdk2.parameters.String("YT phits tmp prefix", required=True)
        chunk_num = sdk2.parameters.Integer("Number of chunks to generate", default=1, required=True)
        releaseTo = sdk2.parameters.String("Release attribute value", required=False)
        ttl = sdk2.parameters.Integer("TTL for released chunks (days, always; 0 for inf)", default=None, required=False)

    def on_enqueue(self):
        # Устанавливаем семафор, имя которого зависит от phits_type.
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_GENERATION_NAME_TEMPLATE.format(type=self.Parameters.advq_phits_type),
                                   capacity=8)
            ],
        )
        return super(AdvqGenPhitsIndex, self).on_enqueue()

    def on_execute(self):
        import yt.wrapper as yt
        import yt.logger as yt_logger

        yt_logger.LOGGER.setLevel(logging.DEBUG)

        if self.Parameters.chunk_num <= 0:
            raise TaskFailure("Chunk_num must be positive, but {!r} found".format(self.Parameters.chunk_num))

        setup_yt_from_parameters(self.Parameters)

        advq_phits_binaries_data = ResourceData(self.Parameters.advq_build_binaries)
        phits_db_generator_binary_path = str(advq_phits_binaries_data.path.joinpath(ADVQ_PHITS_DB_GENERATOR))

        output_prefix = "phits_{type}_{db}_{epd}".format(
            type=self.Parameters.advq_phits_type, db=self.Parameters.advq_db, epd=self.Parameters.advq_epodate)
        output_chunk_indices = list(range(1, self.Parameters.chunk_num + 1))
        resources = []
        output_filenames = []
        phits_resource_class = get_phits_resource_class(self.Parameters.advq_phits_type)
        for i in output_chunk_indices:
            chunk_filename = "{}.{}.{}.db".format(output_prefix, i, self.Parameters.chunk_num)
            res = phits_resource_class(
                self,
                description="Phits chunk {}/{} for {}".format(i, self.Parameters.chunk_num, output_prefix),
                path=chunk_filename,
                advq_phits_type=self.Parameters.advq_phits_type,
                advq_db=self.Parameters.advq_db,
                advq_date=self.Parameters.advq_date,
                advq_epoch=self.Parameters.advq_epoch,
                advq_epodate=self.Parameters.advq_epodate,
                advq_chunk=i,
                advq_total_chunks=self.Parameters.chunk_num,
                advq_is_delta=self.Parameters.advq_is_delta,
                binaries_arcadia_revision=self.Parameters.advq_build_binaries.arcadia_revision,
            )
            if self.Parameters.releaseTo:
                res.released = self.Parameters.releaseTo
            if self.Parameters.ttl is not None:
                res.ttl = convert_ttl(self.Parameters.ttl)
            resources.append(res)
            output_filenames.append(str(res.path))

        env = get_yt_env_from_parameters(self.Parameters)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-db-generate")) as pl:
            cmd = [phits_db_generator_binary_path] + [
                self.Parameters.input_table,
                self.Parameters.phits_tmp_prefix,
            ] + output_filenames
            gen_process = None
            retcode = None
            try:
                gen_process = sp.Popen(cmd, stdin=sp.PIPE, stderr=pl.stdout, env=env)

                retcode = gen_process.wait()
            except Exception:
                if gen_process is not None:
                    gen_process.terminate()
                raise

            if retcode is not None and retcode != 0:
                raise sp.CalledProcessError(retcode, cmd)
            # else
            # Создаём ресурсы.
            for res in resources:
                ResourceData(res).ready()
