# -*- coding: utf-8 -*-
"""
Генерируем phits-таблицы для поддерживаемых баз на YT из входных данных разных поддерживаемых типов.

Таск автоопределяет то, какие данные доступны для генерации.  Через вызовы утилит ADVQ
он создаёт таблицы на YT (в один поток), не создавая и не релизя никаких ресурсов.

При запуске утилит берётся семафор с именем advq_phits_table-{type}.
"""
import json
import logging
import tempfile

from sandbox import sdk2
import sandbox.sdk2.helpers
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import PHITS_GENERATOR
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV, PHITS_TABLES_MINIMAL_REV
from sandbox.projects.advq.common.parameters import PhitsParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

SEMAPHORE_NAME_TEMPLATE = 'advq_phits_table-{type}'

YT_CLOUD_MINIMAL_REV = 4307531  # limit for phits generation only


class AdvqGenPhitsTables(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        disk_space = 3 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        legacy_output = sdk2.parameters.Bool("Legacy, mrkit-compatible output")
        advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")
        limit = sdk2.parameters.Integer("Limit of tables to generate single task run", default=5)
        use_yt_cloud = sdk2.parameters.Bool("Use YT Cloud (not for prod)", required=False, default=False)

    class Context(sdk2.Task.Context):
        to_generate = []

    def on_enqueue(self):
        # Устанавливаем семафор, имя которого зависит от source_type.
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_TEMPLATE.format(type=self.Parameters.advq_phits_type),
                                   capacity=1)
            ],
        )
        return super(AdvqGenPhitsTables, self).on_enqueue()

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [YT_MINIMAL_REV, PHITS_TABLES_MINIMAL_REV])

        advq_phits_binaries_data = ResourceData(self.Parameters.advq_build_binaries)

        env = get_yt_env_from_parameters(self.Parameters)

        if self.Parameters.legacy_output:
            legacy_output_args = ['--legacy-output']
        else:
            legacy_output_args = []

        if self.Parameters.use_yt_cloud:
            validate_arcadia_rev(self.Parameters.advq_build_binaries, [YT_CLOUD_MINIMAL_REV])
            use_yt_cloud_args = ['--with-yt-cloud']
        else:
            use_yt_cloud_args = []

        if self.Parameters.advq_yt_prefix:
            config_file = tempfile.NamedTemporaryFile()
            import yaml
            yaml.safe_dump({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            }, config_file.file)
            config_file.flush()
            env['ADVQ_CONFIG_FILE'] = config_file.name

        phits_generator_binary_path = str(advq_phits_binaries_data.path.joinpath(PHITS_GENERATOR))
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-list-sources")) as pl:
            list_json_str = sp.check_output(
                [phits_generator_binary_path, 'list-gen', self.Parameters.advq_phits_type],
                timeout=600, stderr=pl.stdout, env=env)
            self.Context.to_generate = json.loads(list_json_str)
            if self.Parameters.limit:
                self.Context.to_generate = self.Context.to_generate[:self.Parameters.limit]

        failed = []
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-generate")) as pl:
            for to_generate_args in self.Context.to_generate:
                self.set_info("Generating {}".format(' '.join(to_generate_args)))
                res = sp.Popen(
                    [phits_generator_binary_path, 'gen'] + legacy_output_args + use_yt_cloud_args + [
                        self.Parameters.advq_phits_type] + to_generate_args,
                    stdout=pl.stdout, stderr=sp.STDOUT, env=env
                ).wait()
                if res != 0:
                    failed.append(to_generate_args + [res])
        if failed:
            err_msg = "Failed: {!r}".format(failed)
            self.set_info(err_msg, do_escape=True)
            raise TaskFailure(err_msg)
