# -*- coding: utf-8 -*-
"""
Таск для группировки spikes phits для последующего деплоя в Няню.

Из списка ресурсов с нужными параметрами он пытается сформировать набор phits из 30-минуток за два дня
(в дальнейшем -- подобрать и подходящий sumhits), затем формирует из них новый ресурс и релизит его в Няню.

Дельты не поддерживаются и не нужны.

Ресурс состоит из:
1. Набора phits.
2. (в будущем) одного sumhits.
3. Конфигурационного файла для ADVQ с перечислением файлов.

Обычно spikes за два дня занимают около 30Гб.  Но релизиться это будет каждые полчаса-час...
"""
import json
import logging
import shutil
import tempfile

import pathlib2

import sandbox.common.types.task as ctt
from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import SANDBOX_GROUPER
from sandbox.projects.advq.common.configs import ADVQ_CONFIG_NAME, get_phits_config_class_by_type
from sandbox.projects.advq.common.parameters import AdvqBuildBinariesResourceParameter, releaseTo_params, convert_ttl
from sandbox.projects.common.nanny import nanny
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import Resource, ResourceData

SEMAPHORE_NAME_TMPL = 'advq_grouper_{type}'


class AdvqGroupSpikesDatabases(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        period_size = sdk2.parameters.Integer("Number of consecutive databases", default=48)
        advq_phits_type = sdk2.parameters.String("Phits type", default='spikes')  # на всякий случай
        # пока у spikes другой нет; но вообще нужно поддерживать набор баз
        advq_db = sdk2.parameters.String("Database", default='rus')
        release_new_resource, releaseTo = releaseTo_params()
        advq_build_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation resource", required=True)
        sandbox_api_url = sdk2.parameters.Url("URL of Sandbox API for testing")
        advq_lock_memory = sdk2.parameters.Bool("Lock memory")
        advq_populate = sdk2.parameters.Bool("Populate")
        hosts = sdk2.parameters.List("Server URLs, e.g. http://back00z.advq.yandex.ru:9580/")
        add_databases = sdk2.parameters.Bool("Add databases, not only config (should be used for spikes)")
        with_sumhits = sdk2.parameters.Bool("With sumhits", default=True)
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for 'inf')", default=7, required=True)

    class Context(sdk2.Task.Context):
        do_release = False

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_TMPL.format(type=self.Parameters.advq_phits_type),
                                   capacity=1)
            ]
        )
        super(AdvqGroupSpikesDatabases, self).on_enqueue()

    def on_execute(self):
        if self.Context.do_release:
            with self.memoize_stage.wait_for_parent:
                raise sdk2.WaitTask(
                    tasks=[self.parent],
                    statuses=(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                    wait_all=True)

            # Self-called release task.
            if self.parent.status == ctt.Status.SUCCESS:
                self.server.release(task_id=int(self.parent),
                                    type=self.Parameters.releaseTo,
                                    subject='Automatic release of ' + self.Parameters.description)

        else:
            binaries = ResourceData(self.Parameters.advq_build_binaries)
            grouper_binary_path = str(binaries.path.joinpath(SANDBOX_GROUPER))

            advq_dbs = sorted([self.Parameters.advq_db])  # в будущем будет поддержка нескольких баз.

            if self.Parameters.sandbox_api_url:
                import yaml
                advq_config = {
                    'SANDBOX_API_URL': self.Parameters.sandbox_api_url
                }
                config_file = tempfile.NamedTemporaryFile(prefix='advq_config.')
                yaml.safe_dump(advq_config, config_file.file)
                config_file.file.flush()
                logging.info("Generating config file: %r for SANDBOX_API_URL: %s",
                             config_file.file, self.Parameters.sandbox_api_url)
                env = {
                    'ADVQ_CONFIG_FILE': config_file.name
                }
                logging.info("Using Sandbox API %r through config file %r",
                             self.Parameters.sandbox_api_url, config_file.name)
            else:
                env = {}

            if self.Parameters.release_new_resource:
                variable_args = ['--released', self.Parameters.releaseTo]
            else:
                variable_args = []

            if self.Parameters.advq_lock_memory:
                variable_args.append('--lock-memory')

            if self.Parameters.advq_populate:
                variable_args.append('--populate')

            # If Server list is empty in UI, hosts field contains None, not empty list
            for host in self.Parameters.hosts or []:
                variable_args.extend(['--host', host])

            if self.Parameters.with_sumhits:
                variable_args.append('--with-sumhits')

            # Run grouper util
            with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-chrono-db-generate")) as pl:
                config_json_str = sp.check_output(
                    [grouper_binary_path] + variable_args + [
                        self.Parameters.advq_phits_type, self.Parameters.advq_db, str(self.Parameters.period_size)],
                    timeout=600,
                    stderr=pl.stdout,
                    env=env,
                )
            config_json = json.loads(config_json_str)

            if config_json:
                # Save config as resource
                config_class = get_phits_config_class_by_type(self.Parameters.advq_phits_type)
                data_dir = pathlib2.Path().joinpath('data')
                data_dir.mkdir()
                config_path = data_dir.joinpath(ADVQ_CONFIG_NAME)
                config_path.write_bytes(json.dumps(config_json['advq_config']))

                if self.Parameters.add_databases:
                    # Качаем релизы заново, чтобы выкатить их в составе нового ресурса.
                    for res_id in config_json['resources']:
                        old_resource = Resource[res_id]
                        logging.info("Downloading %r", old_resource)
                        old_resource_data = ResourceData(old_resource)
                        logging.info("Downloaded %r to %r", old_resource, old_resource_data.path)

                        new_path = data_dir.joinpath(old_resource_data.path.name)
                        shutil.copy(str(old_resource_data.path), str(new_path))

                config_res = config_class(
                    task=self,
                    path=str(data_dir),
                    description=("ADVQ config for {phits_type}, dbs={dbs},"
                                 " from {start_date} to {end_date}").format(
                        phits_type=self.Parameters.advq_phits_type,
                        dbs=repr(advq_dbs),
                        start_date=config_json['advq_start_date'],
                        end_date=config_json['advq_end_date'],
                    ),
                    advq_start_date=config_json['advq_start_date'],
                    advq_end_date=config_json['advq_end_date'],
                    advq_period_size=self.Parameters.period_size,
                    advq_resource_count=len(config_json['resources']),
                    advq_dbs=','.join(advq_dbs),
                    advq_phits_type=self.Parameters.advq_phits_type,
                    advq_resources=json.dumps(config_json['resources'])
                )
                config_resource_data = ResourceData(config_res)
                config_resource_data.ready()

                if self.Parameters.release_new_resource:
                    logging.info("Self-releasing")
                    AdvqGroupSpikesDatabases(
                        parent=self,
                        release_new_resource=self.Parameters.release_new_resource,
                        releaseTo=self.Parameters.releaseTo,
                        do_release=True,
                    ).enqueue()
            else:
                logging.info("Nothing to release yet.")

    def on_release(self, parameters):
        nanny.ReleaseToNannyTask2.on_release(self, parameters)
        sdk2.Task.on_release(self, parameters)
        self.mark_released_resources(parameters['release_status'], ttl=convert_ttl(self.Parameters.ttl))
