# -*- coding: utf-8 -*-
import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.projects.advq.artifacts import HITLOG_GENERATOR
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.common.parameters import AdvqBuildBinariesResourceParameter, YtParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters


MIN_HITLOG_CLEANER_ARCADIA_REVISION = 3549131  # ADVQ-1789 keep hitlog tables for 1 year


class AdvqHitlogCleaner(sdk2.Task):
    """
    Удаление старых hitlog таблиц.
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1 * 1024
        ram = 128

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        advq_build_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation utils resource", required=True)
        keep_hitlog_tables_count = sdk2.parameters.Integer(
            'Keep given number of days hitlog tables, starting from most recent',
            required=False, default=None)

        with sdk2.parameters.Group('Debugging') as debugging_group:
            dry_run = sdk2.parameters.Bool('Dry run (i.e. do not remove any table)', default=False)
            verbose_logging = sdk2.parameters.Bool('Verbose logging', default=False)
            advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [MIN_HITLOG_CLEANER_ARCADIA_REVISION, YT_MINIMAL_REV])

        binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)
        args = [str(binaries.path.joinpath(HITLOG_GENERATOR)), 'clean']
        if self.Parameters.keep_hitlog_tables_count is not None:
            args.extend(['--keep-count', str(self.Parameters.keep_hitlog_tables_count)])
        if self.Parameters.dry_run:
            args.append('--dry-run')
        if self.Parameters.verbose_logging:
            args.insert(1, '--verbose')  # --verbose должен быть перед clean, т.к. это общий ключ для всех команд

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters))

        if self.Parameters.advq_yt_prefix:
            config_file = tempfile.NamedTemporaryFile()
            import yaml
            yaml.safe_dump({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            }, config_file.file)
            config_file.flush()
            env['ADVQ_CONFIG_FILE'] = config_file.name

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            sdk2.helpers.subprocess.check_call(args, env=env, stdout=pl.stdout, stderr=pl.stderr)
