# -*- coding: utf-8 -*-
"""
Генерация phits-индекса из отдельно взятой таблицы, в т.ч. в формате mrkit (legacy).  Поддерживает создание нескольких
чанков.
"""
import logging

from sandbox import sdk2
import sandbox.sdk2.helpers
import sandbox.common.types.client as ctc
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import ADVQ_PHITS_DB_GENERATOR, MRKIT_LEGACY_READER
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

from sandbox.projects.advq.common import get_phits_resource_class, validate_arcadia_rev, MRKIT_WITH_DEBUG_REV
from sandbox.projects.advq.common.parameters import PhitsParameters

SEMAPHORE_GENERATION_NAME_TEMPLATE = 'advq_phits_db_generation_single_{type}'


class AdvqLegacyGenPhitsIndex(sdk2.Task):
    """
    Гененрируем чанки phits-индекса для одной legacy phits-таблицы.
    """
    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        cores = 1
        disk_space = 20 * 1024
        ram = 8 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        advq_db = sdk2.parameters.String("Database name", required=True)
        date = sdk2.parameters.String("Date", required=True)
        epoch = sdk2.parameters.Integer("epoch", required=True)
        epodate = sdk2.parameters.String("Date and, perhaps, epoch, formatted as string", required=True)
        input_table = sdk2.parameters.String("YT input table path", required=True)
        chunk_num = sdk2.parameters.Integer("Number of chunks to generate", default=1, required=True)
        ttl = sdk2.parameters.Integer("TTL for released chunks (days, always; 0 for inf)", default=60, required=True)

    def on_enqueue(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [MRKIT_WITH_DEBUG_REV])

        # Устанавливаем семафор, имя которого зависит от phits_type.
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_GENERATION_NAME_TEMPLATE.format(type=self.Parameters.advq_phits_type),
                                   capacity=8)
            ],
        )
        return super(AdvqLegacyGenPhitsIndex, self).on_enqueue()

    def on_execute(self):
        if self.Parameters.chunk_num <= 0:
            raise TaskFailure("Chunk_num must be positive, but {!r} found".format(self.Parameters.chunk_num))

        advq_phits_binaries_data = ResourceData(self.Parameters.advq_build_binaries)
        phits_db_generator_binary_path = str(advq_phits_binaries_data.path.joinpath(ADVQ_PHITS_DB_GENERATOR))
        mrkit_legacy_reader_binary_path = str(advq_phits_binaries_data.path.joinpath(MRKIT_LEGACY_READER))

        output_prefix = "phits_{type}_{db}_{epd}".format(
            type=self.Parameters.advq_phits_type, db=self.Parameters.advq_db, epd=self.Parameters.epodate)
        output_chunk_indices = list(range(1, self.Parameters.chunk_num + 1))
        resources = []
        output_filenames = []
        phits_resource_class = get_phits_resource_class(self.Parameters.advq_phits_type)
        for i in output_chunk_indices:
            chunk_filename = "{}.{}.{}.db".format(output_prefix, i, self.Parameters.chunk_num)
            res = phits_resource_class(
                self,
                description="Phits chunk {}/{} for {}".format(i, self.Parameters.chunk_num, output_prefix),
                path=chunk_filename,
                advq_phits_type=self.Parameters.advq_phits_type,
                advq_db=self.Parameters.advq_db,
                advq_date=self.Parameters.date,
                advq_epoch=self.Parameters.epoch,
                advq_epodate=self.Parameters.epodate,
                advq_chunk=i,
                advq_total_chunks=self.Parameters.chunk_num,
                advq_is_delta=('-delta' in self.Parameters.epodate),
                binaries_arcadia_revision=self.Parameters.advq_build_binaries.arcadia_revision,
            )
            resources.append(res)
            output_filenames.append(str(res.path))

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-phits-db-generate")) as pl:
            channel_args = [
                '--region-field', 'RegionHits:all',
                '--region-field', 'PhoneHits:phone',
                '--region-field', 'TabletHits:tablet',
            ]
            reader_cmd = [mrkit_legacy_reader_binary_path, '--debug', '--json-output', self.Parameters.input_table]
            gen_cmd = [phits_db_generator_binary_path] + channel_args + [
                "-",  # Read from stdin
            ] + output_filenames

            config_env = get_yt_env_from_parameters(self.Parameters)

            gen_process = None
            reader_process = None
            gen_retcode = None
            reader_retcode = None
            try:
                reader_process = sp.Popen(reader_cmd, stdout=sp.PIPE, stderr=pl.stdout, env=config_env)
                gen_process = sp.Popen(gen_cmd, stdin=reader_process.stdout, stderr=pl.stdout)
                gen_retcode = gen_process.wait()
                reader_retcode = reader_process.wait()
            except Exception:
                if reader_process is not None:
                    reader_process.terminate()
                if gen_process is not None:
                    gen_process.terminate()
                raise

            if reader_retcode is not None and reader_retcode != 0:
                raise sp.CalledProcessError(reader_retcode, reader_cmd)
            if gen_retcode is not None and gen_retcode != 0:
                raise sp.CalledProcessError(gen_retcode, gen_cmd)
            # else
            # Создаём ресурсы.
            for res in resources:
                ResourceData(res).ready()
