# -*- coding: utf-8 -*-
import json
import logging
import sandbox.common.types.task as ctt
import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.advq.artifacts import SUMHITS_GENERATOR, ADVQ_SUMHITS_DB_GENERATOR
from sandbox.projects.advq.common import AdvqBroadmatchDatabases, get_sumhits_resource_class, BROADMATCH_RENORM_DICT_FILENAME
from sandbox.projects.advq.common.parameters import PhitsParameters, releaseTo_params, convert_ttl
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.resource import ResourceData
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.task import WaitTask


class AdvqLegacyGenSumhitsIndex(sdk2.Task):
    """
    Создаём sumhits из заданных таблиц.
    Запускается пока без семафора.
    """

    class Requirements(sdk2.Requirements):
        environments = (
            PipEnvironment("yandex-yt"),
            PipEnvironment("yandex-yt-yson-bindings-skynet")
        )
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        disk_space = 3 * 1024
        ram = 1 * 1024
        cores = 1

        class Caches(sdk2.Requirements.Caches):
            pass

    class Context(sdk2.Context):
        do_release = False

    class Parameters(PhitsParameters):
        dbname = sdk2.parameters.String("ADVQ dbname", required=True, default='rus')
        total_chunks = sdk2.parameters.Integer("Number of chunks to generate", required=True, default=1)

        release_new_resource, releaseTo = releaseTo_params()
        period_size = sdk2.parameters.Integer("Number of days/hours/30mins", required=True)
        bnorm_resource = sdk2.parameters.LastReleasedResource("ADVQ broadmatch database",
                                                              resource_type=AdvqBroadmatchDatabases)
        input_sumhits_table = sdk2.parameters.String("YT path to sumhits table", default=None, required=True)
        input_fullqueries_table = sdk2.parameters.String("YT path to fullqueries table",
                                                         required=True)
        epodate = sdk2.parameters.String("Epodate", required=True)
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for inf)", default=30, required=True)

    def on_execute(self):
        if self.Context.do_release:
            # "child-releases-parent pattern"
            with self.memoize_stage.wait_for_parent:
                raise WaitTask([self.parent], statuses=(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                               wait_all=True)
            if self.parent.status == ctt.Status.SUCCESS:
                self.server.release(task_id=int(self.parent),
                                    type=self.Parameters.releaseTo,
                                    subject='Automatic release of ' + self.Parameters.description)
            return

        advq_binaries_resource_path = ResourceData(self.Parameters.advq_build_binaries)

        if self.Parameters.bnorm_resource:
            bnorm_resource_path = ResourceData(self.Parameters.bnorm_resource)
            bnorm_args = ['--bnorm-db-path', str(bnorm_resource_path.path.joinpath(BROADMATCH_RENORM_DICT_FILENAME))]
        else:
            bnorm_args = []

        cmd = [
            str(advq_binaries_resource_path.path.joinpath(SUMHITS_GENERATOR)),
            'gen_index',
            '--advq-generate-sumhits-index-binary', str(advq_binaries_resource_path.path.joinpath(
                ADVQ_SUMHITS_DB_GENERATOR
            )),
            '--index-chunks', str(self.Parameters.total_chunks),
            '--input-sumhits-table', self.Parameters.input_sumhits_table,
            '--input-fullqueries-table', self.Parameters.input_fullqueries_table,
        ] + bnorm_args + [
            self.Parameters.advq_phits_type, self.Parameters.dbname, self.Parameters.epodate,
        ]

        env = get_yt_env_from_parameters(self.Parameters)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-sumhits-gen-index")) as pl:
            data_str = sp.check_output(cmd, stderr=pl.stdout, env=env)
            data = json.loads(data_str)
            resource_class = get_sumhits_resource_class(self.Parameters.advq_phits_type)
            for idx, file in enumerate(data['chunks'], 1):
                res = resource_class(
                    self,
                    description="Sumhits db chunk {}".format(file),
                    path=file,
                    advq_phits_type=self.Parameters.advq_phits_type,
                    advq_db=self.Parameters.dbname,
                    advq_date=data['date'],
                    advq_epoch=data['epoch'],
                    advq_epodate=data['epodate'],
                    advq_start_date=data['start_date'],
                    advq_last_date=data['last_date'],
                    advq_is_delta=False,
                    # TODO(monoid) get default period size from generator
                    period_size=self.Parameters.period_size,
                    advq_total_chunks=self.Parameters.total_chunks,
                    advq_chunk=idx,
                    binaries_arcadia_revision=self.Parameters.advq_build_binaries.arcadia_revision
                )
                ResourceData(res).ready()

        # auto-release
        if self.Parameters.releaseTo:
            # Вызываем сами себя с 'do_release': True.
            logging.info("Self-releasing")
            self.__class__(
                self,
                do_release=True,
                **dict(self.Parameters)).enqueue()

    def on_release(self, parameters):
        super(AdvqLegacyGenSumhitsIndex, self).on_release(parameters)
        self.mark_released_resources(parameters['release_status'], ttl=convert_ttl(self.Parameters.ttl))
