# coding: utf-8

import logging
import os
import tempfile
from contextlib import contextmanager

import yaml

from sandbox import sdk2
from sandbox.common.types.task import Semaphores

from sandbox.projects.advq.artifacts import ADVQ_PHITS_ANTIFRAUD
from sandbox.projects.advq.common import YT_MINIMAL_REV, PHITS_TABLES_MINIMAL_REV, validate_arcadia_rev
from sandbox.projects.advq.common.parameters import (
    PhitsParameters,
    log_verbosity_radio_group,
    get_log_verbosity_flag,
)
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters


SEMAPHORE_NAME = 'advq_antifrod_cleaner'


@contextmanager
def custom_advq_config(env, config):
    if config:
        with tempfile.NamedTemporaryFile() as config_file:
            yaml.safe_dump(
                config,
                config_file.file,
            )
            config_file.flush()
            env['ADVQ_CONFIG_FILE'] = config_file.name
    yield


class AdvqPhitsAntifraudCleaner(sdk2.Task):
    """
    По данным от антифрода строим дельты для phits
    """

    class Requirements(sdk2.Requirements):
        cores = 1
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=SEMAPHORE_NAME,
                    capacity=1,
                )
            ]
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        advq_phits_dir_yt_path = sdk2.parameters.String(
            (
                'Phits path. Actual only for legacy tables and should be set '
                'to //home/advq/advq/{dbname}/phits'
            ),
            default='//home/advq/advq/{dbname}/phits',
            required=False,
        )
        extra_advq_phits_dir_yt_path = sdk2.parameters.String(
            (
                'Extra phits path to lookup tables. For legacy tables should be set '
                'to //home/advq/advq/{dbname}/phits_archive'
            ),
            default='//home/advq/advq/{dbname}/phits_archive',
            required=False,
        )
        new_delta_path = sdk2.parameters.String(
            'Generated deltas YT-path (for testing)',
            required=False,
        )
        processed_banned_hits_path = sdk2.parameters.String(
            'Processed banned hits file YT-path (for testing)',
            required=False,
        )
        advq_db = sdk2.parameters.String("db (rus, tur, etc)", required=True)
        num_days_to_process = sdk2.parameters.Integer(
            'Number of days to process (upper limit)',
            required=False,
            default=None,
        )
        legacy_output = sdk2.parameters.Bool(
            'Legacy (yamr) YT output format',
            required=False,
            default=True,
        )
        log_verbosity = log_verbosity_radio_group(['DEBUG'])

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [YT_MINIMAL_REV, PHITS_TABLES_MINIMAL_REV])

        binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)
        args = [str(binaries.path.joinpath(ADVQ_PHITS_ANTIFRAUD))]

        log_verbosity_flag = get_log_verbosity_flag(self.Parameters.log_verbosity)
        if log_verbosity_flag:
            args.append(log_verbosity_flag)

        if self.Parameters.new_delta_path:
            args.extend([
                '--new-delta-path',
                str(self.Parameters.new_delta_path),
            ])
        if self.Parameters.processed_banned_hits_path:
            args.extend([
                '--processed-banned-hits-path',
                str(self.Parameters.processed_banned_hits_path),
            ])
        if self.Parameters.num_days_to_process:
            args.extend([
                '--num-days-to-process',
                str(self.Parameters.num_days_to_process),
            ])
        if self.Parameters.legacy_output:
            args.append('--legacy-output')

        if self.Parameters.extra_advq_phits_dir_yt_path:
            args.extend((
                '--extra-phits-path',
                str(self.Parameters.extra_advq_phits_dir_yt_path),
            ))

        args.extend((
            '--phits-type', str(self.Parameters.advq_phits_type),
            '--db', str(self.Parameters.advq_db),
        ))

        env = dict(os.environ)

        advq_config_vars = {}
        if self.Parameters.advq_phits_dir_yt_path:
            advq_config_vars['ADVQ_PHITS_DIR_YT_PATH'] = self.Parameters.advq_phits_dir_yt_path

        with custom_advq_config(env=env, config=advq_config_vars):

            env.update(get_yt_env_from_parameters(self.Parameters))

            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
                sdk2.helpers.subprocess.check_call(args, env=env, stdout=pl.stdout, stderr=pl.stderr)
