# -*- coding: utf-8 -*-
import logging
import os

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.projects.advq.artifacts import ADVQ_SUMHITS_CLEANER
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.common.parameters import PhitsParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters


class AdvqSumhitsCleaner(sdk2.Task):
    """
    Удаление старых yt-таблиц sumhits.
    """

    class Requirements(sdk2.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY
        cores = 1
        disk_space = 1 * 1024
        ram = 128

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        advq_db = sdk2.parameters.String('ADVQ dbname', required=False)
        keep_phits_tables_count = sdk2.parameters.Integer(
            'Keep given number of periods phits tables, starting from most recent',
            required=False, default=None)
        keep_sumhits_tables_count = sdk2.parameters.Integer(
            'Keep given number of periods sumhits tables, starting from most recent',
            required=False, default=None)
        keep_queries_count = sdk2.parameters.Integer(
            'Keep given number of periods {full,dyn,diff,real}queries tables, starting from most recent',
            required=False, default=None)
        keep_fullqueries_count = sdk2.parameters.Integer('Keep all sumhits parts for freshest N fullqueries',
                                                         default=3, required=True)

        with sdk2.parameters.Group('Debugging') as debugging_group:
            dry_run = sdk2.parameters.Bool('Dry run (i.e. do not remove any table)', default=False)
            verbose_logging = sdk2.parameters.Bool('Verbose logging', default=False)

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [YT_MINIMAL_REV])

        binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)
        args = [
            str(binaries.path.joinpath(ADVQ_SUMHITS_CLEANER)),
            '--phits-type', str(self.Parameters.advq_phits_type),
            '--keep-fullqueries-count', str(self.Parameters.keep_fullqueries_count),
        ]
        if self.Parameters.advq_db:
            args.extend(['--advq-db', str(self.Parameters.advq_db)])
        if self.Parameters.keep_sumhits_tables_count is not None:
            args.extend(['--keep-tables-count', str(self.Parameters.keep_sumhits_tables_count)])
        if self.Parameters.keep_queries_count is not None:
            args.extend(['--keep-queries-count', str(self.Parameters.keep_queries_count)])
        if self.Parameters.keep_phits_tables_count is not None:
            args.extend(['--keep-phits-tables-count', str(self.Parameters.keep_phits_tables_count)])
        if self.Parameters.dry_run:
            args.append('--dry-run')
        if self.Parameters.verbose_logging:
            args.append('--verbose')

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            sdk2.helpers.subprocess.check_call(args, env=env, stdout=pl.stdout, stderr=pl.stderr)
