# -*- coding: utf-8 -*-
import logging
import tempfile
from sandbox import sdk2
from sandbox.projects.advq.artifacts import SUMHITS_GENERATOR
from sandbox.projects.advq.common.parameters import PhitsParameters
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.resource import ResourceData
from sandbox.sdk2.helpers import subprocess as sp


SEMAPHORE_NAME_TEMPLATE = 'advq_sumhits_dynqueries_table-{type}'


class AdvqSumhitsGenDynqueries(sdk2.Task):
    class Parameters(PhitsParameters):
        parallel = sdk2.parameters.Integer("Generation jobs number", default=4)
        advq_dbs = sdk2.parameters.List("Database names", default=['rus'])
        advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")

    def on_enqueue(self):
        # Устанавливаем семафор, имя которого зависит от source_type.
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_TEMPLATE.format(type=self.Parameters.advq_phits_type),
                                   capacity=1)
            ],
        )
        return super(AdvqSumhitsGenDynqueries, self).on_enqueue()

    def on_execute(self):
        advq_generation_binaries_data = ResourceData(self.Parameters.advq_build_binaries)

        sumhits_generator_binary_path = str(advq_generation_binaries_data.path.joinpath(SUMHITS_GENERATOR))

        yt_env = get_yt_env_from_parameters(self.Parameters)
        env = {}

        if self.Parameters.advq_yt_prefix:
            yt_env.update({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            })
        config_file = tempfile.NamedTemporaryFile()
        import yaml
        yaml.safe_dump(yt_env, config_file.file)
        config_file.flush()
        env['ADVQ_CONFIG_FILE'] = config_file.name

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-sumhits-gen-dynqueries")) as pl:
            for dbname in self.Parameters.advq_dbs:
                sp.check_call(
                    [sumhits_generator_binary_path, 'gen_auto_dynqueries',
                     '--parallel', str(self.Parameters.parallel),
                     self.Parameters.advq_phits_type, dbname],
                    stderr=pl.stdout, env=env)
