# -*- coding: utf-8 -*-
import logging
import tempfile
import os
from sandbox import sdk2
from sandbox.projects.advq.artifacts import SUMHITS_GENERATOR, ADVQ_SUMHITS_FROM_PHITS, ADVQ_SUMHITS_MERGE_HITS
from sandbox.projects.advq.common import validate_arcadia_rev
from sandbox.projects.advq.common.parameters import PhitsParameters
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.resource import ResourceData
from sandbox.sdk2.helpers import subprocess as sp

SEMAPHORE_NAME_TEMPLATE = 'advq_sumhits_gen_table-{type}'

MIN_CHRONO_BUILD_REVISION = 3437238


class AdvqSumhitsGenTables(sdk2.Task):
    class Parameters(PhitsParameters):
        parallel = sdk2.parameters.Integer("Generation jobs number", default=4)
        advq_dbs = sdk2.parameters.List("Database names", default=['rus'])
        advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)")
        use_yt_cloud = sdk2.parameters.Bool("Use YT Cloud (not for production)", default=False, required=True)
        # It is safe to enable this option even for phits_types that do not support chrono.
        with_chrono = sdk2.parameters.Bool("Build also chrono", default=True, required=True)
        with_delta_chrono = sdk2.parameters.Bool("Build chrono epochs as deltas", default=True, required=True)

    class Requirements(sdk2.Task.Requirements):
        cores = 32

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_enqueue(self):
        # Устанавливаем семафор, имя которого зависит от source_type.
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_NAME_TEMPLATE.format(type=self.Parameters.advq_phits_type),
                                   capacity=1)
            ],
        )
        return super(AdvqSumhitsGenTables, self).on_enqueue()

    def on_execute(self):
        advq_generation_binaries_data = ResourceData(self.Parameters.advq_build_binaries)
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [MIN_CHRONO_BUILD_REVISION])

        sumhits_generator_binary_path = str(advq_generation_binaries_data.path.joinpath(SUMHITS_GENERATOR))

        yt_env = get_yt_env_from_parameters(self.Parameters)
        env = dict(os.environ)

        if self.Parameters.advq_yt_prefix:
            yt_env.update({
                'ADVQ_PREFIX': self.Parameters.advq_yt_prefix
            })
        config_file = tempfile.NamedTemporaryFile()
        import yaml
        yaml.safe_dump(yt_env, config_file.file)
        config_file.flush()
        env['ADVQ_CONFIG_FILE'] = config_file.name

        if self.Parameters.with_chrono:
            chrono_args = ['--generate-chrono']
        else:
            chrono_args = []

        if self.Parameters.with_delta_chrono:
            chrono_args.append('--chrono-delta-output')

        if self.Parameters.use_yt_cloud:
            yt_cloud_args = ['--use-yt-cloud']
        else:
            yt_cloud_args = []

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-sumhits-gen-dynqueries")) as pl:
            for dbname in self.Parameters.advq_dbs:
                cmd = [
                    sumhits_generator_binary_path, 'gen_auto_sumhits',
                    '--advq-sumhits-from-phits-binary', str(advq_generation_binaries_data.path.joinpath(
                        ADVQ_SUMHITS_FROM_PHITS
                    )),
                    '--advq-sumhits-merge-hits-binary', str(advq_generation_binaries_data.path.joinpath(
                        ADVQ_SUMHITS_MERGE_HITS
                    )),
                    '--parallel', str(self.Parameters.parallel)
                ] + chrono_args + yt_cloud_args + [
                    self.Parameters.advq_phits_type,
                    dbname
                ]
                sp.check_call(cmd, stderr=pl.stdout, env=env)
