
# -*- coding: utf-8 -*-
# pylint: disable=invalid-name
# pylint: disable=missing-docstring
# pylint: disable=too-few-public-methods
"""
Переносит сгенерированные на sandbox индексы phits на yt
Удаляет старые и не актуальные индексы
"""
import logging
import os

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.artifacts import ADVQ_PHITS_INDEX_UPLOADER
from sandbox.projects.advq.common.parameters import phits_type_radio_group, YtParameters, \
    AdvqBuildBinariesResourceParameter, releaseTo_selector
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters


SEMAPHORE_NAME_TEMPLATE = 'AdvqUploadPhitsIndexToYt-{phits_type}'

# ADVQ-1971: --latest-deltas-count N option and --clean-deltas flag were added.
PHITS_UPLOADER_MINIMAL_REV = 4953969


class AdvqUploadPhitsIndexToYt(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.IPV6
        cores = 1
        disk_space = 16 * 1024  # Minimum disk space in MiB

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        advq_build_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation utils resource", required=True)
        yt = YtParameters
        advq_phits_type = phits_type_radio_group('Source data type', required=True)
        yt_destination_path_template = sdk2.parameters.String(
            'YT destination path template. Should contain {phits_type} and {dbname}.', default=None)
        advq_databases = sdk2.parameters.List('Advq databases', default=[])
        limit = sdk2.parameters.Integer("Resource limit to upload", default=None)
        latest_deltas_count = sdk2.parameters.Integer("Number of latest deltas per period to upload or keep",
                                                      default=None)
        clean_deltas = sdk2.parameters.Bool("Clean superseded deltas (takinng into account Number of latest deltas)")
        releaseTo = releaseTo_selector(
            "Select Sandbox resources with 'released' user attribute value equal to given value")
        with sdk2.parameters.Group('Debugging') as debugging_group:
            dry_run = sdk2.parameters.Bool('Dry run (i.e. do not copy or remove any table)', default=False)
            verbose_logging = sdk2.parameters.Bool('Verbose logging', default=True)

    def on_enqueue(self):
        template = self.Parameters.yt_destination_path_template
        if template:
            try:
                template.format(phits_type='', dbname='')
            except ValueError:
                raise Exception(
                    'yt_destination_path_template should contain {phits_type} and {dbname}.')
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=SEMAPHORE_NAME_TEMPLATE.format(phits_type=self.Parameters.advq_phits_type),
                    capacity=1),
            ],
        )

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_build_binaries, [YT_MINIMAL_REV, PHITS_UPLOADER_MINIMAL_REV])
        binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)

        args = [
            str(binaries.path.joinpath(ADVQ_PHITS_INDEX_UPLOADER)),
            '--phits-type', str(self.Parameters.advq_phits_type),
            '--s-tron', str(self.synchrophazotron)
        ]
        if self.Parameters.advq_databases:
            args.extend(['--advq-db', ','.join(self.Parameters.advq_databases)])
        if self.Parameters.yt_destination_path_template:
            args.extend(['--yt-path-tmpl', self.Parameters.yt_destination_path_template])
        if self.Parameters.limit is not None:
            args.extend(['--limit', str(self.Parameters.limit)])
        if self.Parameters.latest_deltas_count is not None and self.Parameters.latest_deltas_count != 0:
            args.extend(['--latest-deltas-count', str(self.Parameters.latest_deltas_count)])

            if self.Parameters.clean_deltas:
                args.append('--clean-deltas')
        elif self.Parameters.clean_deltas:
            self.set_info("WARNING: Set Number of latest deltas to make Clean deltas effective.")
        if self.Parameters.releaseTo != 'any':
            args.extend(['--sandbox-released', str(self.Parameters.releaseTo)])
        if self.Parameters.dry_run:
            args.append('--dry-run')
        if self.Parameters.verbose_logging:
            args.append('--verbose')

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters))

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            sdk2.helpers.subprocess.check_call(args, env=env, stdout=pl.stdout, stderr=pl.stderr)
