# -*- coding: utf-8 -*-
"""
Обновляем (или скачиваем новое) техническое геодерево ADVQ, сохраняя
как ресурс ADVQ_GEOTREE.

В случае невалидного дерева (по результатам проверок) переходим в FAILURE,
так что может быть полезно в scheduler установить Retry on failure.
"""
from sandbox import sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common.parameters import releaseTo_params, convert_ttl
from sandbox.projects.resource_types import advq_releasers
from sandbox.projects.resource_types.releasers import direct_sre
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.platform_api import platform_api
from sandbox.common.errors import TaskFailure
import sandbox.common.types.resource as ctr
import sandbox.common.types.client as ctc

import logging
import requests
import difflib

from sandbox.sdk2.resource import ResourceData
from .geotree import parse_json_geotree, make_geotree_txt, InvalidGeotreeError


class AdvqGeotree(sdk2.Resource):
    """
    Geotree in ADVQ plain text form.
    """
    releasable = True
    any_arch = True
    auto_backup = True
    ttl = 7  # ресурс легко пересчитывается.
    releasers = direct_sre + advq_releasers


SEMAPHORE_GET_ADVQ_TREE = 'advq_get_advq_tree'


class GetAdvqGeotree(nanny.ReleaseToNannyTask2, platform_api.DeployToPlatformTask, sdk2.Task):
    """
    Скачивание текстового технического геодерева из геобазы,
    преобразование в формат геодерева ADVQ, и сохранение в ресурс типа
    ADVQ_GEOTREE.
    """
    PLATFORM_STABLE_ENVIRONMENTS = [
        'advq.java-video.production',
        'advq.mediareach.production',
        'advq.mediareach.production-backtesting',
        'advq.mediareach.test',
        'advq.mediareach.test-backtesting',
        # ADVQ-1897 stop spikes generation:
        # 'advq.java-spikes.prod-platform',
    ]
    PLATFORM_TESTING_ENVIRONMENTS = [
        # ADVQ-1897 stop spikes generation:
        # 'advq.java-spikes.test',
    ]

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.IPV6
        cores = 1
        disk_space = 3 * 1024
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_GET_ADVQ_TREE,
                                   capacity=1)
            ]
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        geoexport_techtree_url = sdk2.parameters.Url(
            "Full URL for fetching ADVQ geotree",
            default='http://geoexport.yandex.ru/direct-tech-tree?format=json&types=_all_&fields=id,parent,children',
            required=True,
        )
        output_filename = sdk2.parameters.String(
            "Output filename",
            default='geotree.txt',
            required=True,
        )
        compare_with_previous = sdk2.parameters.Bool(
            "Create new resource only if it differs from recent version",
            default=True,
            required=True,
        )

        release_new_resource, releaseTo = releaseTo_params()

        release_to_nanny = sdk2.parameters.Bool(
            "Release to Nanny",
            default=True,
            required=True,
        )
        release_to_platform = sdk2.parameters.Bool(
            "Release to Platform",
            default=True,
            required=True,
        )
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for 'inf')",
                                      default=7, required=True)

    class Context(sdk2.Task.Context):
        has_new_resource = False

    # platform_api.DeployToPlatformTask

    @platform_api.DeployToPlatformTask.oauth_token.getter
    def oauth_token(self):
        return sdk2.Vault.data(self.owner, "platform-oauth-token")

    def get_stable_environments(self):
        return self.PLATFORM_STABLE_ENVIRONMENTS

    def get_testing_environments(self):
        return self.PLATFORM_TESTING_ENVIRONMENTS

    # platform_api.DeployToPlatformTask

    def on_execute(self):
        # Input args
        geotree_url = self.Parameters.geoexport_techtree_url
        output_filename = self.Parameters.output_filename
        compare_with_previous = self.Parameters.compare_with_previous
        release_to = self.Parameters.releaseTo

        # Current size of tech geotree is 4.1M
        json_req = requests.get(geotree_url)
        json_req.raise_for_status()
        json_response = json_req.json()
        try:
            advq_geotree_data = parse_json_geotree(json_response)
        except InvalidGeotreeError as ex:
            raise TaskFailure(ex.message)
        advq_geotree_data_txt = make_geotree_txt(advq_geotree_data)

        if compare_with_previous:
            # Find previous variant to compare
            if self.Parameters.release_new_resource:
                res_attrs = {'released': self.Parameters.releaseTo}
            else:
                res_attrs = {}
            r = sdk2.Resource.find(resource_type=AdvqGeotree, state=ctr.State.READY, attrs=res_attrs).first()
            if r is not None:
                previous_data = ResourceData(r)
                previous_data_txt = previous_data.path.read_bytes()
                if previous_data_txt == advq_geotree_data_txt:
                    logging.info("Geotree didn't change, exiting.")
                    return
                else:
                    # Report difference
                    diff = '\n'.join(difflib.unified_diff(
                        previous_data_txt.split('\n'),
                        advq_geotree_data_txt.split('\n'),
                        lineterm='',  # because .split leaves no newline
                    ))
                    if len(diff) > 128 * 1024:
                        self.set_info("Truncated diff:\n" + diff[:16 * 1024], do_escape=True)
                    else:
                        self.set_info("Diff:\n" + diff, do_escape=True)

        res = AdvqGeotree(
            self,
            description=self.Parameters.description,
            path=output_filename,
            ttl=convert_ttl(self.Parameters.ttl),
        )
        if release_to:
            res.released = release_to

        resource_data = sdk2.ResourceData(res)
        resource_data.path.write_bytes(advq_geotree_data_txt)
        resource_data.ready()
        self.Context.has_new_resource = True

        logging.info("Saved new geotree")

    def on_success(self, prev_status):
        if self.Context.has_new_resource and self.Parameters.releaseTo:
            additional_parameters = dict(
                releaser=self.author,
                release_status=self.Parameters.releaseTo,
                release_subject='Automatic release of ' + self.Parameters.description,
                email_notifications=dict(to=[], cc=[]),
            )
            if self.Parameters.release_to_nanny:
                nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
            if self.Parameters.release_to_platform:
                platform_api.DeployToPlatformTask.on_release(self, additional_parameters)
            self.mark_released_resources(
                self.Parameters.releaseTo,
                ttl=convert_ttl(self.Parameters.ttl),
            )
        sdk2.Task.on_success(self, prev_status)
