# -*- coding: utf-8 -*-
from sandbox import sdk2

from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common.parameters import releaseTo_params, convert_ttl
from sandbox.projects.resource_types import advq_releasers
from sandbox.projects.resource_types.releasers import direct_sre
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.platform_api import platform_api

from sandbox.sdk2.resource import ResourceData
from .stopwords import get_words, DEFAULT_UNIWORD_URL


SEMAPHORE_GET_ADVQ_STOPWORDS = 'advq_get_advq_stopwords'


class AdvqStopwords(sdk2.Resource):
    """
    Stopwords list in plain text form (UTF-8).
    """
    releasable = True
    any_arch = True
    auto_backup = True
    ttl = 7  # ресурс легко пересчитывается.
    releasers = direct_sre + advq_releasers


class GetAdvqStopwords(nanny.ReleaseToNannyTask2, platform_api.DeployToPlatformTask, sdk2.Task):
    PLATFORM_STABLE_ENVIRONMENTS = [
        'advq.java-spikes.prod-platform',
        'advq.java-video.production',
        'advq.mediareach.production',
    ]
    PLATFORM_TESTING_ENVIRONMENTS = [
        'advq.java-spikes.test',
        'advq.mediareach.test',
    ]

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=SEMAPHORE_GET_ADVQ_STOPWORDS,
                                   capacity=1)
            ]
        )

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        uniword_url = sdk2.parameters.Url(
            "Full URL for fetching Uniword table",
            default=DEFAULT_UNIWORD_URL,
            required=True,
        )
        output_filename = sdk2.parameters.String(
            "Output filename",
            default='stopwords',
            required=True,
        )
        compare_with_previous = sdk2.parameters.Bool(
            "Create new resource only if it differs from recent version",
            default=True,
            required=True,
        )

        release_new_resource, releaseTo = releaseTo_params()

        release_to_nanny = sdk2.parameters.Bool(
            "Release to Nanny",
            default=True,
            required=True,
        )
        release_to_platform = sdk2.parameters.Bool(
            "Release to Platform",
            default=True,
            required=True,
        )
        ttl = sdk2.parameters.Integer("TTL for released resource (days, always; 0 for 'inf')",
                                      default=7, required=True)

    class Context(sdk2.Task.Context):
        has_new_resource = False

    # platform_api.DeployToPlatformTask

    @platform_api.DeployToPlatformTask.oauth_token.getter
    def oauth_token(self):
        return sdk2.Vault.data(self.owner, "platform-oauth-token")

    def get_stable_environments(self):
        return self.PLATFORM_STABLE_ENVIRONMENTS

    def get_testing_environments(self):
        return self.PLATFORM_TESTING_ENVIRONMENTS

    # platform_api.DeployToPlatformTask

    def on_execute(self):
        advq_stopwords_data_txt = '\n'.join(get_words(self.Parameters.uniword_url))
        res = AdvqStopwords(
            self,
            description=self.Parameters.description,
            path=self.Parameters.output_filename,
            ttl=convert_ttl(self.Parameters.ttl),
        )
        resource_data = ResourceData(res)
        resource_data.path.write_bytes(advq_stopwords_data_txt)
        resource_data.ready()
