# -*- coding: utf-8 -*-
"""
Создает базу Advquick как ресурс в sandbox. Заменяет собой задачу GetAdvquickDatabase
1. берётся самая свежая таблица sumhits rus, с максимальной эпохой,
2. из нее берутся поля  OrigSanitized и  Hits с помощью mrkit_legacy_reader,
3. сохраняются в текстовом формате с разделением полей табом, без заголовка, текст сжимается  gzip.
4. Сейчас в имени выходного файла нет ни даты ни эпохи. Дата и эпоха сохраняются в аттрибутах ресурса
"""
import logging
import os
from datetime import date
from pipes import quote

import sandbox.common.types.client as ctc
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.resource import State
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.artifacts import MRKIT_LEGACY_READER
from sandbox.projects.advq.common import SHELL_COMMAND_PREFIX
from sandbox.projects.advq.common.list_res import list_sumhits_tables
from sandbox.projects.advq.common.parameters import releaseTo_params, convert_ttl, PhitsParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters, setup_yt_from_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import ADVQUICK_DATABASE
from sandbox.sandboxsdk.environments import PipEnvironment

GET_ADVQUICK_DATABASE2_SEMAPHORE = "GET_ADVQUICK_DATABASE2_SEMAPHORE"
RESOURCE_ROOT_DIR = 'dbs'


class GetAdvquickDatabase2(nanny.ReleaseToNannyTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.IPV6
        cores = 1
        disk_space = 16 * 1024
        environments = [
            PipEnvironment("yandex-yt"),
        ]

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(PhitsParameters):
        advq_db = sdk2.parameters.String('Database name', required=True, default='rus')
        min_db_size = sdk2.parameters.Integer(
            'Output database minimum size for release', required=True, default=1024,
            description='database with size < min_db_size (bytes) will not be released (simple check for db validity)'
        )
        only_new = sdk2.parameters.Bool(
            'Keep existing sandbox resources', required=True, default=True,
            description='Create output database only if there is no sb resource with specified date'
        )
        output_file = sdk2.parameters.String('Output filename', required=True, default='akitaregs.txt.gz')
        release_new_resource, releaseTo = releaseTo_params()
        ttl = sdk2.parameters.Integer("TTL for released resources (days, always; 0 for inf)", default=60, required=True)

        with sdk2.parameters.Group('Debugging') as debugging_group:
            dry_run = sdk2.parameters.Bool('Dry run (i.e. do not create any resource)', default=False)
            advq_yt_prefix = sdk2.parameters.String("YT prefix instead of //home/advq (for testing proposes)",
                                                    default=None)

    class Context(sdk2.Task.Context):
        has_new_resource = False

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=GET_ADVQUICK_DATABASE2_SEMAPHORE,
                    capacity=1)
            ]
        )

    def on_execute(self):
        setup_yt_from_parameters(self.Parameters)

        # найти самую свежую sumhits таблицу в виде tuple (date, epoch, table_name)
        (latest_date, epoch, sumhits_table) = get_latest_sumhits_table(self.Parameters.advq_phits_type, self.Parameters.advq_db)
        if not sumhits_table:
            raise TaskError('There are no sumhits tables in {}'.format(str(self.Parameters.yt_path)))
        self.set_info('Found most recent sumhits table: {}'.format(sumhits_table))
        # проверить существует ли ресурс для даты latest_date из tuple
        if self.Parameters.only_new:
            search_attrs = {'date': latest_date.isoformat()}
            if self.Parameters.release_new_resource:
                search_attrs['released'] = str(self.Parameters.releaseTo)
            advquick_db_resource = ADVQUICK_DATABASE.find(attrs=search_attrs, state=State.READY).first()
            if advquick_db_resource:
                self.set_info('Resource with date {} already exists: {}'.format(latest_date, str(advquick_db_resource)))
                # это не ошибка, поэтому просто выходим, но ничего не релизим, т.к. уже есть база за такую дату
                return

        # сгенерить файл
        self.set_info('Generating output table')
        binaries = sdk2.ResourceData(self.Parameters.advq_build_binaries)
        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters))

        os.mkdir(RESOURCE_ROOT_DIR)
        db_name = os.path.join(RESOURCE_ROOT_DIR, str(self.Parameters.output_file))

        cmd = "{mrkit_legacy_reader} {sumhits_table} OrigSanitized Hits | gzip -9c >{output_filename}" \
            .format(
                mrkit_legacy_reader=quote(str(binaries.path.joinpath(MRKIT_LEGACY_READER))),
                sumhits_table=quote(sumhits_table),
                output_filename=quote(db_name)
            )
        shell_cmd = SHELL_COMMAND_PREFIX + [cmd]

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            sdk2.helpers.subprocess.check_call(shell_cmd, env=env, stdout=pl.stdout, stderr=pl.stderr)

        db_size = os.path.getsize(db_name)
        if db_size <= self.Parameters.min_db_size:
            raise TaskError("Resulting resource size is too small {}".format(db_size))

        # создать ресурс
        if not self.Parameters.dry_run:
            self.set_info('Creating Sandbox resource')
            advquick_db_resource = ADVQUICK_DATABASE(self,
                                                     description='Advquick database for date {}'.format(latest_date),
                                                     path=db_name)
            # установить аттрибуты
            advquick_db_resource.date = latest_date.isoformat()
            advquick_db_resource.epoch = str(epoch)
            # зарелизить ресурс
            if self.Parameters.release_new_resource:
                advquick_db_resource.released = str(self.Parameters.releaseTo)
                self.Context.has_new_resource = True

            sdk2.ResourceData(advquick_db_resource).ready()

    def on_success(self, prev_status):
        if self.Context.has_new_resource and self.Parameters.release_new_resource:
            additional_parameters = dict(
                releaser=self.author,
                release_status=self.Parameters.releaseTo,
                release_subject='Automatic release of ' + self.Parameters.description,
                email_notifications=dict(to=[], cc=[]),
            )
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
            self.mark_released_resources(additional_parameters['release_status'], ttl=convert_ttl(self.Parameters.ttl))
        sdk2.Task.on_success(self, prev_status)


def get_latest_sumhits_table(phits_type, advq_db):
    # type: (str, str) -> (date, int, str)
    """
    находит и возвращает самую свежую sumhits таблицу (с максимальной датой и максимальной эпохой для этой даты).
    таблицы с неправильными именами игнорируются. Если ничего не найдено, вернет (None, None, None)
    :param phits_type:
    :param advq_db:
    :return: tuple(date, epoch, base_table_name)
    """
    max_date_epoch_table = (None, None, None)
    for date_epoch_table in list_sumhits_tables(phits_type, advq_db):
        max_date_epoch_table = max(max_date_epoch_table, date_epoch_table)
    # парсим дату, если она есть
    date_str, epoch, full_path = max_date_epoch_table
    if date_str:
        date_value = date(int(date_str[:4]), int(date_str[4:6]), int(date_str[6:8]))
        max_date_epoch_table = (date_value, epoch, full_path)
    return max_date_epoch_table
