from __future__ import print_function
import logging
import os
from datetime import datetime, timedelta
import sandbox.common.types.client as ctc
import sandbox.common.types.resource as ctr
from sandbox.projects.advq.common import list_res, PokazometerGenerationBinaries
from sandbox import sdk2
from sandbox.projects.advq.artifacts import (
    POKAZOMETER_BUILD_DB, POKAZOMETER_CLICK_GENERATOR, POKAZOMETER_SHOW_GENERATOR
)
from sandbox.projects.advq.common.yt_utils import setup_yt_from_parameters, get_yt_env_from_parameters
from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import POKAZOMETER_DATABASE_CLICKS, POKAZOMETER_DATABASE_SHOWS
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common.parameters import releaseTo_params, convert_ttl, YtParameters
from sandbox.sdk2 import ResourceData

GET_POKAZOMETER_SEMAPHORE = 'advq_get_pokazometer_database'


def gen_config(
        config_path,
        clicks,
        shows,
        adnetevents_yt_path,
        stopword_path,
        phrases_path,
        workdir,
        dbs_dir,
):
    with open(config_path, 'wb') as out:
        print("[Generator]", file=out)
        print("update_clicks = {}".format(int(clicks)), file=out)
        print("update_shows = {}".format(int(shows)), file=out)
        print("adnetevents_yt_dir = {}".format(adnetevents_yt_path), file=out)
        print("workdir = {}".format(workdir), file=out)
        print("dbsdir = {}".format(dbs_dir), file=out)
        print("phrases = {}".format(phrases_path), file=out)
        print("stopwords = {}".format(stopword_path), file=out)


class PokazometerStopwordList(sdk2.Resource):
    """ Pokazometer stopword list resource """
    ttl = 'inf'
    auto_backup = True


class GetPokazometerDatabase2(nanny.ReleaseToNannyTask2, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.IPV6
        cores = 2
        disk_space = 16 * 1024  # 16 GB
        ram = 256 * 1024  # 256 GB
        environments = [
            PipEnvironment("yandex-yt"),
        ]
        semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=GET_POKAZOMETER_SEMAPHORE,
                    capacity=1)
            ]
        )

    class Caches(sdk2.Requirements.Caches):
        pass

    class Context(sdk2.Context):
        has_new_resource = False

    class Parameters(sdk2.Parameters):
        # clicks or shows
        # binaries
        adnetevents_yt_path = sdk2.parameters.String(
            "adnetevents dir YT path",
            default="//home/advq/advq/adnetevents",
        )
        stopwords_resource = sdk2.parameters.Resource(
            "Pokazometer stopword list",
            resource_type=PokazometerStopwordList,
            required=True,
        )
        pokazometer_generation_binaries_resource = sdk2.parameters.Resource(
            "Pokazometer generation binaries",
            resource_type=PokazometerGenerationBinaries,
            required=True,
        )
        yt_params = YtParameters()
        ttl = sdk2.parameters.Integer("TTL for released resources (days, always; 0 for inf)", default=60, required=True)
        release_new_resource, releaseTo = releaseTo_params()
        with sdk2.parameters.RadioGroup("Database type", required=True) as db_type:
            db_type.values["clicks"] = db_type.Value(value="clicks")
            db_type.values["shows"] = db_type.Value(value="shows")

    def on_execute(self):
        setup_yt_from_parameters(self.Parameters.yt_params)

        binaries_path = sdk2.ResourceData(self.Parameters.pokazometer_generation_binaries_resource).path
        stopwords_path = sdk2.ResourceData(self.Parameters.stopwords_resource).path

        config_path = "./pokazometer.conf"
        workdir = "."
        phrases_path = "./phrases"
        dbs_dir = "./dbs"

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters.yt_params))

        adnetevents_dates = list_res.list_any_tables(self.Parameters.adnetevents_yt_path)
        max_date_str, _ = max(adnetevents_dates)
        # Previously "%Y-%m-%d" was used for res attr, but it's just one generation to change this to ADVQ-consistent
        max_date = datetime.strptime(max_date_str, "%Y%m%d").date()

        db_type = self.Parameters.db_type

        if db_type == 'clicks':
            res_class = POKAZOMETER_DATABASE_CLICKS

            # It's a weired logic of original pokazometer: if you provide Monday, it will
            # generate index for previous Sunday (and use Sunday in the dir name) :(
            # There is a room for improvement.
            max_date_monday = max_date - timedelta(days=max_date.weekday())
            max_date_sunday = max_date_monday - timedelta(days=1)
            # Previously "%Y-%m-%d" was used for res attr, but it's just one generation to change this
            max_date_monday_str = max_date_monday.strftime("%Y%m%d")
            max_date_sunday_str = max_date_sunday.strftime("%Y%m%d")

            attrs = {
                'date': max_date_monday_str
            }
            if self.Parameters.release_new_resource:
                attrs['released'] = self.Parameters.releaseTo
            res = sdk2.Resource.find(resource_type=res_class, attrs=attrs, state=ctr.State.READY).first()
            if res:
                self.set_info("Clicks database for {} is already built".format(max_date_monday_str))
            else:
                gen_config(
                    config_path,
                    clicks=True,
                    shows=False,
                    adnetevents_yt_path=self.Parameters.adnetevents_yt_path,
                    stopword_path=str(stopwords_path),
                    phrases_path=phrases_path,
                    workdir=workdir,
                    dbs_dir=dbs_dir,
                )
                cmd = [
                    str(binaries_path.joinpath(POKAZOMETER_BUILD_DB)),
                    '--date', max_date_monday_str,
                    '--click-generator-binary', str(binaries_path.joinpath(POKAZOMETER_CLICK_GENERATOR)),
                    '--show-generator-binary', str(binaries_path.joinpath(POKAZOMETER_SHOW_GENERATOR)),
                    '--pokazometer-config-path', config_path,
                ]
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
                    sdk2.helpers.subprocess.check_call(cmd, env=env, stdout=pl.stdout, stderr=pl.stderr)
                self.pack_resource(
                    dbs_path="dbs/clicks_{}".format(max_date_sunday_str),
                    res_class=res_class,
                    out_path="clicks_dbs.tgz",
                    date_str=max_date_monday_str,
                )
                self.Context.has_new_resource = True
        elif db_type == 'shows':
            res_class = POKAZOMETER_DATABASE_SHOWS

            attrs = {
                'date': max_date_str,
            }
            if self.Parameters.release_new_resource:
                attrs['released'] = self.Parameters.releaseTo
            res = sdk2.Resource.find(resource_type=res_class, attrs=attrs, state=ctr.State.READY).first()
            if res:
                self.set_info("Shows database for {} is already built".format(max_date_str))
            else:
                gen_config(
                    config_path,
                    clicks=False,
                    shows=True,
                    adnetevents_yt_path=self.Parameters.adnetevents_yt_path,
                    stopword_path=str(stopwords_path),
                    phrases_path=phrases_path,
                    workdir=workdir,
                    dbs_dir=dbs_dir,
                )
                cmd = [
                    str(binaries_path.joinpath(POKAZOMETER_BUILD_DB)),
                    '--date', max_date_str,
                    '--click-generator-binary', str(binaries_path.joinpath(POKAZOMETER_CLICK_GENERATOR)),
                    '--show-generator-binary', str(binaries_path.joinpath(POKAZOMETER_SHOW_GENERATOR)),
                    '--pokazometer-config-path', config_path,
                ]
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
                    sdk2.helpers.subprocess.check_call(cmd, env=env, stdout=pl.stdout, stderr=pl.stderr)
                self.pack_resource(
                    dbs_path="dbs/shows_{}".format(max_date_str),
                    res_class=res_class,
                    out_path="shows_dbs.tgz",
                    date_str=max_date_str,
                )
                self.Context.has_new_resource = True

    def pack_resource(self, dbs_path, res_class, out_path, date_str):
        # Compress with tar gz
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            cmd = ["tar", "-cvzf", out_path, "-C", dbs_path, "."]
            sdk2.helpers.subprocess.check_call(cmd, stdout=pl.stdout, stderr=pl.stderr)
        # Create resource
        res = res_class(
            self,
            description="Pokazometer database {}".format(out_path),
            path=out_path,
            date=date_str,
            binaries_arcadia_revision=self.Parameters.pokazometer_generation_binaries_resource.arcadia_revision,
        )
        ResourceData(res).ready()

    def on_success(self, prev_status):
        if self.Context.has_new_resource and self.Parameters.releaseTo:
            additional_parameters = dict(
                releaser=self.author,
                release_status=self.Parameters.releaseTo,
                release_subject='Automatic release of ' + self.Parameters.description,
                email_notifications=dict(to=[], cc=[]),
            )
            nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
            self.mark_released_resources(
                self.Parameters.releaseTo,
                ttl=convert_ttl(self.Parameters.ttl),
            )
        sdk2.Task.on_success(self, prev_status)
