# -*- coding: utf-8 -*-
"""
Обработка рекламных фраз директа на YT
"""

import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores

from sandbox.projects.advq.artifacts import OFFLINE_PHITS_EXPORT, OFFLINE_PHITS_UTIL
from sandbox.projects.advq.common import (
    # OfflinePhitsArtifacts,
    validate_arcadia_rev,
    YT_MINIMAL_REV
)
from sandbox.projects.advq.common.parameters import YtParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
# from sandbox.projects.advq.GetAdvqGeotree import AdvqGeotree
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

DIRECT_EXPORT_SEMAPHORE = 'offline_phits_direct_export'

# Bug fixes, new options, etc
MIN_REVISION = 4352662


class OfflinePhitsDirectExport(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 1024
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        artifacts = sdk2.parameters.LastReleasedResource(
            "offline-phits artifacts",
            # type=OfflinePhitsArtifacts,  # FIXME: invalid argument (SANDBOX-6404)
            required=True
        )
        geotree = sdk2.parameters.LastReleasedResource(
            "advq geotree",
            # type=AdvqGeotree,  # FIXME: invalid argument (SANDBOX-6404)
            required=True
        )
        phits_dir = sdk2.parameters.String("ypath to phits databases", required=True)
        out_dir = sdk2.parameters.String("ypath to output", required=True)
        direct_path = sdk2.parameters.String("ypath to direct export table", required=True)
        phits_days = sdk2.parameters.Integer("number of days to forecast", required=True)
        min_phits_days = sdk2.parameters.Integer("minimum number of days for result interpolation", required=True)
        new_pool_size = sdk2.parameters.Integer("pool size for new dates", required=True, default=2)
        old_pool_size = sdk2.parameters.Integer("pool size for old dates", required=True, default=10)
        query_memsize = sdk2.parameters.Integer("memory limit for query processing YT jobs, MB", default=2048)
        use_yt_cloud = sdk2.parameters.Bool("use YT cloud", required=False, default=False)

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(name=DIRECT_EXPORT_SEMAPHORE, capacity=1)
            ],
        )
        return super(OfflinePhitsDirectExport, self).on_enqueue()

    def on_execute(self):
        if self.Parameters.artifacts is None:
            raise TaskFailure("offline-phits artifacts not found")
        if self.Parameters.geotree is None:
            raise TaskFailure("geotree not found")

        validate_arcadia_rev(self.Parameters.artifacts, [YT_MINIMAL_REV, MIN_REVISION])

        artifacts_data = ResourceData(self.Parameters.artifacts)

        binary_path = str(artifacts_data.path.joinpath(OFFLINE_PHITS_EXPORT))
        util_path = str(artifacts_data.path.joinpath(OFFLINE_PHITS_UTIL))
        geotree_path = str(ResourceData(self.Parameters.geotree).path)

        env = get_yt_env_from_parameters(self.Parameters)
        env['YT_USE_CLIENT_PROTOBUF'] = '0'
        env['YT_LOG_LEVEL'] = 'DEBUG'

        if self.Parameters.use_yt_cloud:
            env['YT_SPEC'] = '{"pool_trees":["physical"],"tentative_pool_trees":["cloud"]}'

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("direct-export")) as pl:
            sp.check_call(
                [
                    binary_path, '--yt-server', self.Parameters.yt_proxy, '--phits-dir', self.Parameters.phits_dir,
                    '--out-dir', self.Parameters.out_dir, '--direct-path', self.Parameters.direct_path,
                    '--phits-days', str(self.Parameters.phits_days), '--java-util-path', util_path, '--geotree-path',
                    geotree_path,
                    '--min-phits-days', str(self.Parameters.min_phits_days),
                    '--new-pool-size', str(self.Parameters.new_pool_size),
                    '--old-pool-size', str(self.Parameters.old_pool_size),
                    '--query-memsize', str(self.Parameters.query_memsize),
                ],
                stderr=pl.stdout, env=env)
