# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData

from sandbox.projects.advq.common.parameters import YtParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.projects.advq.artifacts import OFFLINE_PHITS_UPLOAD
from sandbox.projects.advq.common import (
    OfflinePhitsArtifacts,
    validate_arcadia_rev,
    YT_MINIMAL_REV
)

ADVQ_2008_MINIMAL_REV = 5695603


class OfflinePhitsUpload(sdk2.Task):
    class Requirements(sdk2.Requirements):
        cores = 1
        disk_space = 128
        ram = 128

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(YtParameters):
        artifacts = sdk2.parameters.LastReleasedResource(
            "offline-phits artifacts",
            resource_type=OfflinePhitsArtifacts,
            required=True
        )
        dbs_path = sdk2.parameters.String("ypath to directory with new databases", required=True)
        offline_phits_path = sdk2.parameters.String("ypath to offline phits directory", required=True)
        limit = sdk2.parameters.Integer("upload limit in days", required=False)
        yt_ttl = sdk2.parameters.Integer("TTL for YT resources, in days", required=False)

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.artifacts, [YT_MINIMAL_REV, ADVQ_2008_MINIMAL_REV])

        env = get_yt_env_from_parameters(self.Parameters)

        artifacts_data = ResourceData(self.Parameters.artifacts)
        binary_path = str(artifacts_data.path.joinpath(OFFLINE_PHITS_UPLOAD))
        args = []

        if self.Parameters.yt_ttl is not None:
            if self.Parameters.limit is None:
                self.info("WARNING: yt_ttl without limit may lead to cycled copying of expired resources")
            elif self.Parameters.limit >= self.Parameters.yt_ttl:
                self.info("WARNING: yt_ttl smaller than limit may lead to cycled copying of expired resources")
            args.extend(['--yt-ttl', str(self.Parameters.yt_ttl)])
        if self.Parameters.limit is not None:
            args.extend(['--limit', str(self.Parameters.limit)])

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("offline-phits-upload")) as pl:
            sp.check_call(
                [
                    binary_path, '--db-dir', self.Parameters.dbs_path, '--offline-phits-dir',
                    self.Parameters.offline_phits_path
                ] + args,
                stderr=pl.stdout, env=env)
