# -*- coding: utf-8 -*-
"""
Получает список дат, для которых не сгенерированы таблицы,
генерирует таблицы и кладет их в //home/advq/advq/{db}/pkz
"""
import logging
import json

from sandbox import sdk2
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.common import AdvqGenerationBinaries
from sandbox.projects.advq.artifacts import PKZ_GENERATOR
from sandbox.projects.advq.common.parameters import YtParameters
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters

SEMAPHORE_NAME = 'advq_pkz_gen_tables_{db}'


class PkzGenTables(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        advq_generation_binaries = sdk2.parameters.Resource(
            "ADVQ generation resource", resource_type=AdvqGenerationBinaries, required=True)
        yt = YtParameters
        db = sdk2.parameters.String("Database(rus, tur)", required=True)
        days_count = sdk2.parameters.Integer("Days to consider", default=28)
        pkz_tables_yt_prefix = sdk2.parameters.String("YT prefix of pkz tables", default="//home/advq/advq/{db}/pkz")

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=SEMAPHORE_NAME.format(db=self.Parameters.db),
                    capacity=1
                ),
            ],
        )

    def on_execute(self):
        binaries_data = sdk2.ResourceData(self.Parameters.advq_generation_binaries)

        binaries_path = str(binaries_data.path.joinpath(PKZ_GENERATOR))

        env = get_yt_env_from_parameters(self.Parameters)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('advq-pkz-list-gen')) as pl:
            list_json_str = sdk2.helpers.subprocess.check_output([
                binaries_path,
                'list-gen', self.Parameters.db,
                '--days-count', str(self.Parameters.days_count),
                '--output-table', self.Parameters.pkz_tables_yt_prefix],
                env=env, stderr=pl.stdout
            )
            to_generate = json.loads(list_json_str)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(__name__)) as pl:
            for to_generate_args in to_generate:
                sdk2.helpers.subprocess.check_call([
                        binaries_path,
                        'gen',
                        '--output-table',
                        self.Parameters.pkz_tables_yt_prefix.format(db=self.Parameters.db) + '/' + to_generate_args,
                        self.Parameters.db,
                        to_generate_args,
                    ], env=env, stdout=pl.stdout, stderr=pl.stderr
                )
