# -*- coding: utf-8 -*-
"""
Генерируем hitlog-таблицы на YT из RotateHitLog.

Таск автоопределяет то, какие данные доступны для генерации.  Через вызовы утилит ADVQ
он создаёт таблицы на YT, не производя и не релизя никаких ресурсов.

При запуске утилит берётся семафор с именем advq_hitlog_table с счётчиком 1.
"""
import logging
import os

from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.projects.advq.artifacts import OFFLINE_PKZ_MAIN, OFFLINE_PKZ_PROCESS
from sandbox.projects.advq.common import validate_arcadia_rev, YT_MINIMAL_REV
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.types.client as ctc
from sandbox.sdk2.resource import ResourceData

OFFLINE_PKZ_MINIMAL_REV = 4164856


class RunOfflinePkz(sdk2.Task):
    """
    Run offline-pkz
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY & ctc.Tag.IPV6
        cores = 1
        disk_space = 1024
        ram = 256

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        yt_proxy = sdk2.parameters.String("YT server name", required=True)
        yt_token_vault_user = sdk2.parameters.String("YT key vault owner")
        yt_token_vault_name = sdk2.parameters.String("YT key vault name")
        offline_pkz_binaries = sdk2.parameters.LastReleasedResource("Offline PKZ binaries resource", required=True)
        yt_directory = sdk2.parameters.String("YT directory with databases", required=True)
        yt_config_directory = sdk2.parameters.String("YT directory with config", required=False)
        pkz_days = sdk2.parameters.Integer("Number of days", required=True, default=28)
        limit = sdk2.parameters.Integer("Limit number of tables processed", default=None, required=False)
        position = sdk2.parameters.String("Position, format: P{slot}{position}", default="P24", required=False)
        min_slot = sdk2.parameters.Integer("Min slot(1/2)", default=2, required=False)
        max_slot = sdk2.parameters.Integer("Max slot(1/2)", default=2, required=False)

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.offline_pkz_binaries, [YT_MINIMAL_REV, OFFLINE_PKZ_MINIMAL_REV])

        pkz_binaries_data = ResourceData(self.Parameters.offline_pkz_binaries)

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters))

        offline_pkz_main_binary_path = str(pkz_binaries_data.path.joinpath(OFFLINE_PKZ_MAIN))
        offline_pkz_main_process_path = str(pkz_binaries_data.path.joinpath(OFFLINE_PKZ_PROCESS))

        call_args = [
            offline_pkz_main_binary_path,
            '--pkz-yt-folder', str(self.Parameters.yt_directory),
            '--pkz-days', str(self.Parameters.pkz_days),
            '--position', self.Parameters.position,
            '--min-slot', str(self.Parameters.min_slot),
            '--max-slot', str(self.Parameters.max_slot),
            '--offline-pkz-process-binary', offline_pkz_main_process_path,
        ]
        if self.Parameters.yt_config_directory:
            call_args.extend(['--pkz-yt-config-folder', self.Parameters.yt_config_directory])
        if self.Parameters.limit is not None:
            call_args.extend(['--limit', str(self.Parameters.limit)])
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("run_offline_pkz")) as pl:
            try:
                sp.check_call(
                    call_args,
                    stdout=pl.stdout,
                    stderr=sp.STDOUT,
                    env=env
                )
            except subprocess.CalledProcessError as ex:
                self.set_info(ex.message)
                raise
