# -*- coding: utf-8 -*-
"""
Таск для автодеплоя: генерации и выкладки новых конфигов по мере обновления баз.
"""
import six
import json
import logging
import posixpath

import sandbox.common.types.task as ctt
from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.common.errors import TaskFailure
from sandbox.common.types.task import Semaphores
from sandbox.projects.advq.AdvqApiDataTest import AdvqApiDataTest
from sandbox.projects.advq.AdvqDeployConfigToHosts import CONFIG_FILENAME, AdvqDeployConfigToHosts
from sandbox.projects.advq.artifacts import ADVQ_STATS_CONFIG_GENERATOR
from sandbox.projects.advq.autodeploy.AdvqAutodeployHelperGenTestingConfig import AdvqAutodeployHelperGenTestingConfig
from sandbox.projects.advq.autodeploy.AdvqAutodeployHelperGetServiceInfo import AdvqAutodeployHelperGetServiceInfo
from sandbox.projects.advq.autodeploy.common import DEFAULT_PROCESS_TIMEOUT, WAIT_TIMEOUT, \
    MIN_ADVQ_DEPLOY_BINARY, _get_configs, _build_config_res
from sandbox.projects.advq.common import validate_arcadia_rev, AdvqApiDataTestReport
from sandbox.projects.advq.common.configs import AdvqDeployConfiguration
from sandbox.projects.advq.common.parameters import CommonPhitsParameters, NannyTokenParameters, PlatformTokenParameters, \
    releaseTo_selector, SandboxParameters, StartrekParameters
from sandbox.projects.yabs.release.duty.schedule import get_current_engine_responsibles
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.resource import ResourceData, Resource
from sandbox.sdk2.task import WaitTime, WaitTask

SEMAPHORE_NAME_TMPL = 'advq_autodeploy_{type}_{released}'

STARTREK_QUEUE = 'ADVQDEPLOY'
STARTREK_API_URL = "https://st-api.yandex-team.ru/v2/"
STARTREK_UI_URL = "https://st.yandex-team.ru/"


class AdvqAutodeploy(sdk2.Task):
    """
    Generate and deploy ADVQ config with intermediate testing phase.
    """

    class Parameters(sdk2.Task.Parameters):
        nanny_token = NannyTokenParameters
        platform_token = PlatformTokenParameters
        sandbox_parameters = SandboxParameters
        startrek_parameters = StartrekParameters
        phits_parameters = CommonPhitsParameters
        input_data_release = releaseTo_selector("Databases release status", with_empty=True)
        releaseTo = releaseTo_selector("Release configs to")

        new_only = sdk2.parameters.Bool("Release config only if it differs from old one", required=True)

        with_sumhits = sdk2.parameters.Bool("With sumhits", required=True)
        with_sumhits_mini = sdk2.parameters.Bool("With sumhits mini", required=True)
        with_weekly = sdk2.parameters.Bool("With weeklyhits", required=True)
        with_monthly = sdk2.parameters.Bool("With monthlyhits", required=True)

        with with_weekly.value[True]:
            lock_weekly = sdk2.parameters.Bool("Lock weekly databases", required=False, default=False)

        flat_layout = sdk2.parameters.Bool(
            "Flat layout",
            required=True,
            default=False,
            description="With flat layout, each machine has full set of chunks (e.g. for spikes)"
        )

        replication_factor = sdk2.parameters.Integer("Replication factor", required=True, default=1,
                                                     description="Replicate files over given number of hosts")
        clear_tags = sdk2.parameters.Bool("Clear files tags", required=True, default=False,
                                          description="Remove from tags everything, that is not a valid host name")
        max_offline_hosts = sdk2.parameters.Integer(
            "Max offline hosts",
            required=True,
            default=0,  # safe value for phits types without replication
            description="Continue deploy, if number of offline hosts is less than given value"
        )

        raw_platform_hostnames = sdk2.parameters.Bool("Raw platform hostnames")
        with_http = sdk2.parameters.Bool("With HTTP download data (fresh backend required)")
        with_sandbox_ids = sdk2.parameters.Bool("Add Sandbox IDs for useful info in logs (fresh backend required)")

        with sdk2.parameters.Group("Datatest parameters") as datatest_group:
            minimal_check = sdk2.parameters.Bool("Perform only minimal datatest", default=True)
            create_ticket_on_failure = sdk2.parameters.Bool(
                "Create Startreck ticket on datatest failure",
                default=False,
            )
            data_test_startrek_queue = sdk2.parameters.String(
                'Startrek queue for tickets about failed tests',
                default=STARTREK_QUEUE
            )
            data_test_startrek_assign_to_yabs_duty = sdk2.parameters.Bool(
                "Use YABS_DUTY_SCHEDULE resource to assign ticket to people on duty")
            data_test_balancer_url = sdk2.parameters.String("Balancer URL for datatest (random host by default)",
                                                            required=False)

        force_deploy = sdk2.parameters.Bool("Force deploy", description="""Perform deploy even if service is
initially in inconsistent state.
""")

        redeploy = sdk2.parameters.Bool("Redeploy", description="""redeploy config from scratch""", default=False)
        dry_run = sdk2.parameters.Bool("Dry run", description="Do not really release configs")

        deploy_config_res = sdk2.parameters.Resource(
            "Deploy configuration resource JSON",
            resource_type=AdvqDeployConfiguration,
        )
        layout_memory_limit = sdk2.parameters.Float("Memory limit for generated config", required=False)
        layout_disk_limit = sdk2.parameters.Float("Disk limit for generated config", required=False)

        ram_disk_limit = sdk2.parameters.Bool("required disk_free_space > ram_free_space", required=True, default=False)

        service_wait = sdk2.parameters.Integer(
            "Wait time interval between retries for deploy, seconds (60-300sec for spikes, 1200 for normal, etc)",
            default=300)
        get_configs_retries = sdk2.parameters.Integer(
            "Get service retries number",
            default=10)
        nanny_services_with_port = sdk2.parameters.List(
            "Nanny services to deploy to, with possible forced port after semicolon")
        platform_services = sdk2.parameters.List(
            "Platform components to deploy to (<project>.<application>.<environment>.<component>)")

        databases = sdk2.parameters.List(
            "Databases (default list if empty)",
            description="""In any incomprehensible situation, use default list.

Default list depends on phits type and is a list of keys of
advq.generation.common.config.StandardConfig.DATABASES[phits_type]""")
        ttl = sdk2.parameters.Integer("TTL for released config resource (0 for inf)", default=30)

    class Context(sdk2.Task.Context):
        info_prefix = ''
        # get_configs
        get_configs_task_id = None
        service_info_res_id = None
        previous_configs_res_id = None

        testing_config_queue_json = None

        dates_str = 'UNDEFINED'

        nanny_cleaned_services = None

        # Deploy process vars
        preparing_deploy_now = None
        deploying_now = None
        deploy_queue = None
        deployed = None
        deploy_repeat_count = None

        already_tested_deploy_approved = False
        startrek_ticket_id = None
        # счётчик для генерации неповторяющихся директорий для ресурсов
        resource_counter = 0

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024
        environments = (PipEnvironment('startrek_client', '1.7.0', use_wheel=True),)

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_enqueue(self):
        self.Requirements.semaphores = Semaphores(
            acquires=[
                Semaphores.Acquire(
                    name=SEMAPHORE_NAME_TMPL.format(
                        type=self.Parameters.phits_parameters.advq_phits_type,
                        released=self.Parameters.releaseTo,
                    ),
                    capacity=1,
                )
            ],
            # не отпускаем при WAIT!
            release=(
                ctt.Status.Group.BREAK, ctt.Status.Group.FINISH
            )
        )
        super(AdvqAutodeploy, self).on_enqueue()

    def on_execute(self):
        validate_arcadia_rev(
            self.Parameters.phits_parameters.advq_build_binaries,
            [MIN_ADVQ_DEPLOY_BINARY]
        )
        binaries = ResourceData(self.Parameters.phits_parameters.advq_build_binaries)

        if (not self.Parameters.nanny_services_with_port) and (not self.Parameters.platform_services):
            raise TaskFailure("Please, define at least Nanny service or Platform component to deploy to")

        self.Context.max_offline_hosts = int(self.Parameters.max_offline_hosts)
        if self.Context.max_offline_hosts < 0:
            # Special way to force deploy without check below
            self.Context.max_offline_hosts = -self.Context.max_offline_hosts
        elif self.Context.max_offline_hosts != 0 and (
                int(self.Context.max_offline_hosts) > (int(self.Parameters.replication_factor) - 1)) and (
                not self.Parameters.flat_layout):
            # В случае flat layout replication factor фактически равно количеству хостов
            raise TaskFailure("max_offline_hosts parameters should be smaller than replication factor")

        config_generator_path = str(binaries.path.joinpath(ADVQ_STATS_CONFIG_GENERATOR))

        if self.Parameters.dry_run:
            self.Context.info_prefix = 'DRY_RUN: '
        elif self.Parameters.force_deploy:
            self.Context.info_prefix = 'FORCE: '
        else:
            self.Context.info_prefix = ''

        # Получаем информацию о текущем состоянии кластера.
        with self.memoize_stage.launch_get_configs:
            get_service_info_retries = (int(self.Parameters.get_configs_retries) + 1) // 2
            get_service_wait_time = (get_service_info_retries + 1) * int(self.Parameters.service_wait)
            get_service_info_task = AdvqAutodeployHelperGetServiceInfo(
                self,
                description=("Get current service info for {}".format(", ".join(
                    self.Parameters.nanny_services_with_port + self.Parameters.platform_services
                ))),
                nanny_token_vault_user=self.Parameters.nanny_token.nanny_token_vault_user,
                nanny_token_vault_name=self.Parameters.nanny_token.nanny_token_vault_name,
                platform_token_vault_user=self.Parameters.platform_token.platform_token_vault_user,
                platform_token_vault_name=self.Parameters.platform_token.platform_token_vault_name,
                advq_build_binaries=self.Parameters.phits_parameters.advq_build_binaries.id,
                advq_phits_type=self.Parameters.phits_parameters.advq_phits_type,
                releaseTo=self.Parameters.releaseTo,
                raw_platform_hostnames=self.Parameters.raw_platform_hostnames,
                force_deploy=self.Parameters.force_deploy,
                service_wait=self.Parameters.service_wait,
                max_offline_hosts=self.Context.max_offline_hosts,
                get_configs_retries=get_service_info_retries,
                nanny_services_with_port=self.Parameters.nanny_services_with_port,
                platform_services=self.Parameters.platform_services,
            )
            get_service_info_task.enqueue()
            self.Context.get_configs_task_id = get_service_info_task.id
            raise WaitTask(
                [get_service_info_task],
                statuses=(ctt.Status.Group.BREAK + ctt.Status.Group.FINISH),
                timeout=get_service_wait_time,
            )
        with self.memoize_stage.get_service_info(commit_on_entrance=False):
            get_service_info_task = sdk2.Task[
                self.Context.get_configs_task_id]  # type: AdvqAutodeployHelperGetServiceInfo
            if get_service_info_task.status != ctt.Status.SUCCESS:
                raise TaskFailure("GetServiceInfo child task #{!r} in unexpected state {!s}".format(
                    get_service_info_task.id, get_service_info_task.status
                ))

            self.Context.service_info_res_id = get_service_info_task.Parameters.result_service_info_res_id
            self.Context.previous_configs_res_id = get_service_info_task.Parameters.result_previous_configs_res_id
            self.set_info(self.Context.info_prefix + "Got service info: #{!r}, previous configs: #{!r}".format(
                self.Context.service_info_res_id,
                self.Context.previous_configs_res_id,
            ))

        def clean_nanny_service(service_with_port):
            cnt = service_with_port.count(':')
            if cnt == 0:
                return service_with_port
            elif cnt == 1:
                return service_with_port.split(':', 1)[0]
            else:
                raise ValueError("Malformed nanny service id with port: {!r}".format(service_with_port))

        # Создаём конфиг, в котором создаём новую testing ветку.
        with self.memoize_stage.start_gen_testing_config_task:
            self.Context.nanny_cleaned_services = [clean_nanny_service(service_with_port)
                                                   for service_with_port in self.Parameters.nanny_services_with_port]
            if self.Parameters.redeploy and len(self.Parameters.nanny_services_with_port) > 1:
                raise Exception("Deploy from scratch required only one nanny servise.")
            if self.Parameters.redeploy and not self.Parameters.force_deploy:
                raise Exception("force deplot required when redeploy")
            gen_testing_config_task = AdvqAutodeployHelperGenTestingConfig(
                self,
                description=("Generate new testing configs for {}".format(", ".join(
                    self.Parameters.nanny_services_with_port + self.Parameters.platform_services
                ))),
                input_service_info_res=self.Context.service_info_res_id,
                input_previous_configs_res=self.Context.previous_configs_res_id,
                advq_phits_type=self.Parameters.phits_parameters.advq_phits_type,
                advq_build_binaries=self.Parameters.phits_parameters.advq_build_binaries.id,
                input_data_release=self.Parameters.input_data_release,
                with_sumhits=self.Parameters.with_sumhits,
                with_sumhits_mini=self.Parameters.with_sumhits_mini,
                with_weekly=self.Parameters.with_weekly,
                with_monthly=self.Parameters.with_monthly,
                ram_disk_limit=self.Parameters.ram_disk_limit,
                lock_weekly=self.Parameters.lock_weekly,
                flat_layout=self.Parameters.flat_layout,
                new_only=self.Parameters.new_only,
                force_deploy=self.Parameters.force_deploy,
                redeploy=self.Parameters.redeploy,
                raw_platform_hostnames=self.Parameters.raw_platform_hostnames,
                deploy_config_res=self.Parameters.deploy_config_res.id if self.Parameters.deploy_config_res else None,
                layout_memory_limit=self.Parameters.layout_memory_limit,
                layout_disk_limit=self.Parameters.layout_disk_limit,
                databases=self.Parameters.databases,
                nanny_services=self.Context.nanny_cleaned_services,
                platform_services=self.Parameters.platform_services,
                sandbox_token_vault_user=self.Parameters.sandbox_parameters.sandbox_token_vault_user,
                sandbox_token_vault_name=self.Parameters.sandbox_parameters.sandbox_token_vault_name,
                replication_factor=self.Parameters.replication_factor,
                clear_tags=self.Parameters.clear_tags,
                with_http=self.Parameters.with_http,
                with_sandbox_ids=self.Parameters.with_sandbox_ids,
            )
            gen_testing_config_task.enqueue()
            self.Context.gen_testing_config_task_id = gen_testing_config_task.id
            raise WaitTask(
                [gen_testing_config_task],
                statuses=ctt.Status.Group.BREAK + ctt.Status.Group.FINISH
            )
        with self.memoize_stage.gen_new_configs(commit_on_entrance=False):
            gen_testing_config_task = sdk2.Task[
                self.Context.gen_testing_config_task_id]  # type: AdvqAutodeployHelperGenTestingConfig
            if gen_testing_config_task.status != ctt.Status.SUCCESS:
                raise TaskFailure("GenTestingConfig child task #{!r} in unexpected state {!s}".format(
                    gen_testing_config_task.id, gen_testing_config_task.status
                ))
            testing_config_queue_json = gen_testing_config_task.Parameters.result_config_queue_json
            try:
                self.Context.dates_str = json.loads(gen_testing_config_task.Parameters.result_report_json or '{}').get(
                    'dates', 'UNDEFINED'
                )
            except ValueError:
                logging.exception("report parsing failed")

            # Инициализируем контекст для тестового деплоя
            self.Context.deploy_queue = json.loads(testing_config_queue_json)
            self.Context.deploying_now = None
            self.Context.deployed = []

            self.set_info("Generated test config. Deploy queue: {!s}".format(self.Context.deploy_queue))

            # Может оказаться, что deploy_queue пуст -- если не появилось новых чанков и
            # тестовая ветвь совпадает со стабильной.
            if not self.Context.deploy_queue:
                self.set_info(self.Context.info_prefix + "Nothing to deploy, exiting...")
                return

            # проверяем совпадение конфига с уже сгенеренным ранее
            if not gen_testing_config_task.Parameters.result_new_config:
                #   находим последний отчет о тестировании сделанный на текущем конфиге
                config_json = self._get_full_deployed_config_json(self.Context.deploy_queue)
                self.set_info("Test config is the same as previous: {!s}".format(config_json))
                report_res = AdvqApiDataTestReport.find(attrs={'tested_config': config_json}).first()
                if report_res is not None and report_res.startrek_ticket_id is not None:
                    self.set_info("Found previous testing report: #{!r}".format(report_res.id))
                    ticket_key = report_res.startrek_ticket_id
                    startrek = self._get_startrek_client()
                    ticket = startrek.issues[ticket_key]
                    assert ticket.key == ticket_key
                    info_prefix = '{prefix}The configuration is already tested and failed,' \
                                  'ticket <a href="{url}{ticket}">{ticket}</a> in Startrek is ' \
                        .format(prefix=self.Context.info_prefix, url=STARTREK_UI_URL, ticket=ticket_key)

                    if ticket.status.key != 'closed':
                        self.set_info(info_prefix + "still open, there is nothing to do", do_escape=False)
                        return

                    elif ticket.resolution.key != 'fixed':  # тикет закрыт без фикса => форсируем деплой
                        self.set_info(info_prefix + "closed with resolution 'will not fix', forcing deploy",
                                      do_escape=False)
                        self.Context.already_tested_deploy_approved = True

                    else:  # тикет закрыт как исправленный, тесты должны пройти, надо их прогнать и проверить
                        self.set_info(info_prefix + "closed with resolution 'fixed', running tests again",
                                      do_escape=False)
                        self.Context.startrek_ticket_id = ticket_key  # понадобится, если тесты опять упадут
                else:
                    self.set_info("Previous testing report is not found. Will run tests")

        # Мы сгенерировали тестовый конфиг; теперь выкатываем его по очереди на разные сервисы.
        with self.memoize_stage.deploying_testing(commit_on_entrance=False, commit_on_wait=False):
            # в случае форсированного деплоя не делаем тестовую выкатку и тестирование.
            if self.Parameters.force_deploy or self.Parameters.dry_run or self.Context.already_tested_deploy_approved:
                self.Context.deployed = self.Context.deploy_queue
                self.Context.deploy_queue = []
            else:
                self._do_deploy_repeatedly(
                    config_generator_path,
                    subject="Deploying testing config {} to {}, task #{}".format(
                        self.Context.dates_str,
                        self.Parameters.phits_parameters.advq_phits_type,
                        self.id,
                    ),
                )

        # Тестируем, при неудаче завершаем задачу
        if not self.Parameters.force_deploy and not self.Parameters.dry_run:
            self.set_info("Running tests")
            if not self._test_testing_branch():
                return

        # Генерируем релизный конфиг
        with self.memoize_stage.gen_release_configs(commit_on_entrance=False):
            self._gen_release_config_resources(config_generator_path)

        # Выкатываем релизный конфиг.  self.Context был заполнен в _gen_release_config_resources.
        with self.memoize_stage.deploy_release(commit_on_entrance=False, commit_on_wait=False):
            if not self.Parameters.dry_run:
                self._do_deploy_repeatedly(
                    config_generator_path,
                    subject="Deploying release config {} to {}, task #{}".format(
                        self.Context.dates_str,
                        self.Parameters.phits_parameters.advq_phits_type,
                        self.id,
                    ),
                )
        # Генерируем чистый конфиг
        if not self.Parameters.force_deploy:
            with self.memoize_stage.gen_clean_configs(commit_on_entrance=False):
                self._gen_clean_config_resources(config_generator_path)

            # Выкатываем очищеный конфиг.  self.Context был заполнен в _gen_clean_config_resources.
            with self.memoize_stage.deploy_clean(commit_on_entrance=False, commit_on_wait=False):
                if not self.Parameters.dry_run:
                    self._do_deploy_repeatedly(
                        config_generator_path,
                        subject="Deploying clean config {} to {}, task #{}".format(
                            self.Context.dates_str,
                            self.Parameters.phits_parameters.advq_phits_type,
                            self.id,
                        ),
                    )

        self.set_info(self.Context.info_prefix + "Successfully deployed.")

    #
    # Phase 2. Deploying test configs.
    # Phase 5. Deploying release configs.
    # Phase 7. Deploying clean configs.
    #
    def _do_deploy_repeatedly(self, config_generator_path, subject=None):
        """
        Depoloy generated configs to services, one by one.

        It takes pairs (service_id, config_resource_id) from self.Context.deploy_queue, from last
        (so, it is not real queue).  Released services are pushed to self.Context.deployed,
        as pairs (service_id, deployed_config_resource_id).  For some technical reasons
        deployed_config_resource_id is not same as config_resource_id.

        May be used with new testing config, or with release config after test.

        This method repeatedly restarts waiting for timeout or for other task completion.

        :param config_generator_path: path to advq_stats_config_generator
        :return: None
        """
        # self.set_info(self.Context.info_prefix + "Service info res: {!r}".format(self.Context.service_info_res_id))

        assert self.Context.service_info_res_id, repr(self.Context.service_info_res_id)
        service_info_data = ResourceData(Resource[self.Context.service_info_res_id])

        if subject is None:
            subject = "Deploying {} to ADVQ {}".format(
                self.Context.dates_str,
                self.Parameters.phits_parameters.advq_phits_type,
            )
        self.set_info(subject)

        if self.Context.preparing_deploy_now is not None:
            service_id, deploy_task_id = self.Context.preparing_deploy_now
            deploy_task = sdk2.Task[deploy_task_id]  # type: AdvqDeployConfigToHosts
            if deploy_task.status == ctt.Status.SUCCESS:
                # релизим
                self.set_info(self.Context.info_prefix + "Releasing config {} to {}".format(
                    self.Context.dates_str, service_id
                ))
                self.server.release(
                    task_id=deploy_task_id,
                    subject="{} ({!r})".format(subject, service_id),
                    type=self.Parameters.releaseTo,
                )
                raise WaitTask(
                    [deploy_task_id], statuses=tuple(ctt.Status.Group.BREAK) + (
                        ctt.Status.RELEASED, ctt.Status.NOT_RELEASED,
                    ),
                    timeout=WAIT_TIMEOUT,
                )
            elif deploy_task.status == ctt.Status.RELEASED:
                self.set_info(
                    self.Context.info_prefix + "Deployed #{} to {}, waiting for its readiness on hosts".format(
                        deploy_task.Parameters.result_config_resource_id, service_id))
                self.Context.preparing_deploy_now = None
                self.Context.deploying_now = (
                    service_id,
                    deploy_task.Parameters.result_config_resource_id,
                    deploy_task.Parameters.input_resource.id
                )
                self.Context.deploy_repeat_count = int(self.Parameters.get_configs_retries)
            else:
                raise TaskFailure("Child config releasing task {!r} failed".format(deploy_task_id))

        if self.Context.deploying_now is not None:
            # Мы что-то уже деплоим.
            service_id, target_res_id, orig_res_id = self.Context.deploying_now
            # проверяем, что выкатка завершилась.
            config_res_ids = _get_configs(
                # TODO(monoid) здесь опрашиваются все сервисы, нам достаточно одного!
                self,
                config_generator_path,
                service_info_filename=str(service_info_data.path)
            )
            invalid_hosts = [host for host, res_id_or_err in six.iteritems(config_res_ids[service_id])
                             if res_id_or_err != target_res_id]
            if len(invalid_hosts) <= self.Context.max_offline_hosts:
                # Выкатили успешно.
                self.set_info(self.Context.info_prefix + "Deployed {!r} to {!r}{!s}".format(
                    target_res_id,
                    service_id,
                    "" if not invalid_hosts else ", invalid hosts: {}, {}".format(len(invalid_hosts), invalid_hosts)
                ))
                self.Context.deployed.append((service_id, target_res_id))
                self.Context.deploying_now = None
                # проваливаемся вниз, к выкатке на следующий сервис
            else:
                # Ещё не выехало, ждём, если есть ещё попытки
                self.Context.deploy_repeat_count -= 1
                if self.Context.deploy_repeat_count > 0:
                    self.set_info(self.Context.info_prefix + "Waiting for {} to deploy, attempt {}/{}".format(
                        service_id, self.Context.deploy_repeat_count, self.Parameters.get_configs_retries))
                    raise WaitTime(self.Parameters.service_wait)
                else:
                    raise TaskFailure("Failed to get #{!r} deployed to {!r} after several waits".format(
                        target_res_id,
                        service_id,
                    ))

        # Начинаем выкатку на следующий сервис из списка
        while self.Context.deploy_queue and not self.Context.preparing_deploy_now:
            # Выбираем, что релизить.
            service_id, config_res = self.Context.deploy_queue.pop()
            deploy_task = AdvqDeployConfigToHosts(
                parent=self,
                description="{} to {}".format(subject, service_id),
                input_resource=config_res,
                deploy_to_nanny=(service_id in self.Context.nanny_cleaned_services),
                deploy_to_platform=(service_id in self.Parameters.platform_services),
                ttl=self.Parameters.ttl,
            )
            deploy_task.enqueue()
            self.Context.preparing_deploy_now = (service_id, deploy_task.id)
            raise WaitTask([deploy_task], statuses=(ctt.Status.Group.BREAK + ctt.Status.Group.FINISH),
                           timeout=WAIT_TIMEOUT)

    #
    # Phase 3
    #
    # TODO тестирование нужно производить после каждого деплоя теста перед переходом к предыдущему?
    # TODO на тот случай, если новая база из-за ошибок уложит показывающие машины.
    def _test_testing_branch(self):
        """
        returns True on testing success and False on testing failure or raise error if any
        :rtype: bool
        """
        with self.memoize_stage.data_test_start:
            test_balancer_url = self.Parameters.data_test_balancer_url
            if not test_balancer_url:
                # Выбираем случайный хост
                service_info_res_data = ResourceData(Resource[self.Context.service_info_res_id])
                with service_info_res_data.path.open('rb') as si_file:
                    service_info = json.load(si_file)
                import random
                random.seed(self.id)  # воспроизводимая случайность.
                random_testing_hosts = random.choice([
                    info['hosts'] for info in six.itervalues(service_info)
                    if info['hosts']  # пропускаем пустые группы, хотя их быть не должно
                ])
                random_host = random.choice(random_testing_hosts)
                self.set_info(self.Context.info_prefix + "Using host {} for testing".format(random_host),
                              do_escape=True)
                test_balancer_url = 'http://{}/'.format(random_host)
            release_url = posixpath.join(test_balancer_url, 'advq')
            testing_url = posixpath.join(test_balancer_url, 'advq_test')
            data_test_task = AdvqApiDataTest(
                self,
                description="Autotest for #{}".format(self.id),
                advq_build_binaries=self.Parameters.phits_parameters.advq_build_binaries.id,
                test_config=self._get_full_deployed_config_json(self.Context.deployed),
                release_url=release_url,
                testing_url=testing_url,
                # дополнительно юзаем already_tested_deploy_approved,
                # чтобы минимальное тестирование все таки было в любом случае
                minimal_check=self.Parameters.minimal_check or self.Context.already_tested_deploy_approved,
            )
            self.Context.data_test_task_id = data_test_task.id
            self.set_info(self.Context.info_prefix + "Starting data test task #{}, urls {} and {}".format(
                data_test_task.id,
                release_url,
                testing_url,
            ))
            data_test_task.enqueue()
            raise WaitTask(
                [data_test_task],
                statuses=ctt.Status.Group.FINISH + ctt.Status.Group.BREAK,
                wait_all=True,
                timeout=WAIT_TIMEOUT,
            )
        with self.memoize_stage.data_test_finish:
            data_test_task = sdk2.Task[self.Context.data_test_task_id]  # type: AdvqApiDataTest
            if data_test_task.status != ctt.Status.SUCCESS:
                raise TaskFailure("Autotest task #{} execution failed: {}".format(
                    data_test_task.id,
                    data_test_task.status,
                ))
            elif not data_test_task.Parameters.result_success:
                if self.Parameters.create_ticket_on_failure:
                    # создать тикет в стартреке вместо письма,
                    # номер тикета прицепить атрибутом к отчету,
                    # отчет прицепить к тикету
                    # если текущий конфиг уже ранее тестировался и есть закрытый пофикшенный тикет, делаем линк на него
                    report_html_res = Resource[
                        data_test_task.Parameters.result_report_res_id]  # type: AdvqApiDataTestReport
                    report_html_data = ResourceData(report_html_res)
                    startrek = self._get_startrek_client()
                    startrek_args = {}
                    if self.Parameters.data_test_startrek_assign_to_yabs_duty:
                        duty_crew = get_current_engine_responsibles()
                        if duty_crew:
                            # If crew happens to be absent, it's better to create ticket anyway than fail
                            startrek_args['assignee'] = duty_crew[0]
                            startrek_args['followers'] = duty_crew
                    ticket = startrek.issues.create(
                        queue=self.Parameters.data_test_startrek_queue,
                        summary='Autotest task #{} for {} have failed'.format(data_test_task.id,
                                                                              self.Context.dates_str),
                        description=(
                            "Autotest task #((https://sandbox.yandex-team.ru/task/{task_id}/view {task_id})) have failed.\n"
                            "Configuration: {configuration}\n"
                            "Dates: {dates}\n"
                            "See attached test report for details").format(
                            task_id=data_test_task.id,
                            configuration=report_html_res.tested_config,
                            dates=self.Context.dates_str,
                        ), **startrek_args)
                    ticket.attachments.create(str(report_html_data.path))
                    report_html_res.startrek_ticket_id = ticket.key
                    # связываем с тикетом self.Context.startrek_ticket_id, если он есть
                    if self.Context.startrek_ticket_id is not None:
                        ticket.links.create(issue=self.Context.startrek_ticket_id, relationship='relates')
                    self.set_info("Datatest failed, see report in resource #{}".format(
                        data_test_task.Parameters.result_report_res_id
                    ))
                    self.set_info("Created ticket <a href='{url}{ticket}'>{ticket}</a>".format(
                        url=STARTREK_UI_URL, ticket=ticket.key
                    ), do_escape=False)
                else:
                    raise TaskFailure("Datatest failed, see report in resource #{}".format(
                        data_test_task.Parameters.result_report_res_id
                    ))
            return data_test_task.Parameters.result_success
        return True

    #
    # Phase 4.  Generate release configs' resources.
    #
    def _gen_release_config_resources(self, config_generator_path):
        """
        Get services and testing (presumably) configs from self.Context.deployed, and generate
        release config resources, adding them to self.Context.deploy_queue.

        :param config_generator_path: path to advq_stats_config_generator
        :return: None
        """
        deployed = self.Context.deployed
        self.Context.deployed = []
        assert not self.Context.deploying_now, repr(self.Context.deploying_now)
        assert not self.Context.preparing_deploy_now, repr(self.Context.preparing_deploy_now)
        assert not bool(self.Context.deploy_queue), repr(self.Context.deploy_queue)
        if self.Parameters.force_deploy:
            # В случае force_deploy сразу выкатываем очищеный конфиг.
            clean_args = []
        else:
            clean_args = ['--do-not-clean']
        with sandbox.sdk2.helpers.ProcessLog(
            self, logger=logging.getLogger("config_generator_switch_layout")
        ) as pl:
            for service_id, testing_config_id in deployed:
                # Не делаем здесь никакого логгирования, т.к. код просто перелопачивает JSON.
                release_config_json = sp.check_output(
                    [config_generator_path, 'release_testing'] +
                    clean_args +
                    [str(ResourceData(Resource[testing_config_id]).path.joinpath(CONFIG_FILENAME))],
                    timeout=DEFAULT_PROCESS_TIMEOUT,
                    stderr=pl.stdout,
                )
                release_config = json.loads(release_config_json)

                self.Context.deploy_queue.append(
                    (service_id, int(_build_config_res(self, service_id, release_config, "release",
                                                       count=self.Context.resource_counter)))
                )
                self.Context.resource_counter += 1

    #
    # Phase 6.  Generate clean configs' resources.
    #
    def _gen_clean_config_resources(self, config_generator_path):
        """
        Get services and testing (presumably) configs from self.Context.deployed, and generate
        release config resources, adding them to self.Context.deploy_queue.

        :param config_generator_path: path to advq_stats_config_generator
        :return: None
        """
        deployed = self.Context.deployed
        self.Context.deployed = []
        assert not self.Context.deploying_now, repr(self.Context.deploying_now)
        assert not self.Context.preparing_deploy_now, repr(self.Context.preparing_deploy_now)
        assert not bool(self.Context.deploy_queue), repr(self.Context.deploy_queue)
        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("config_generator_switch_layout")) as pl:
            for service_id, testing_config_id in deployed:
                # Не делаем здесь никакого логгирования, т.к. код просто перелопачивает JSON.
                clean_config_json = sp.check_output([
                    config_generator_path,
                    'clean_config',
                    str(ResourceData(Resource[testing_config_id]).path.joinpath(CONFIG_FILENAME))
                ],
                    timeout=DEFAULT_PROCESS_TIMEOUT,
                    stderr=pl.stdout,
                )
                clean_config = json.loads(clean_config_json)

                self.Context.deploy_queue.append(
                    (service_id, int(_build_config_res(self, service_id, clean_config, "clean",
                                                       count=self.Context.resource_counter)))
                )
                self.Context.resource_counter += 1

    def _get_full_deployed_config_json(self, deployed):
        # type: (list) -> str
        """ возвращает полный текущий конфиг.
        Собирает его из deployed и self.Context.previous_configs_res_id
        :return JSON string
        """
        previous_configs_res_data = ResourceData(Resource[self.Context.previous_configs_res_id])
        with previous_configs_res_data.path.open('rb') as pc_file:
            result = json.load(pc_file)

        result.update(deployed)

        return json.dumps(result, sort_keys=True)

    def _get_startrek_client(self):
        parameters = self.Parameters.startrek_parameters
        if not parameters.startrek_token_vault_name:
            raise ValueError("OAuth token for Startrek is missing")

        if parameters.startrek_token_vault_user:
            token = sdk2.Vault.data(parameters.startrek_token_vault_user, parameters.startrek_token_vault_name)
        else:
            token = sdk2.Vault.data(parameters.startrek_token_vault_name)

        from startrek_client import Startrek
        return Startrek(useragent='sandbox-task', token=token, base_url=STARTREK_API_URL)
