# -*- coding: utf-8 -*-
import json
import logging
import tempfile
import six

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.advq.AdvqDeployConfigToHosts import CONFIG_FILENAME
from sandbox.projects.advq.artifacts import ADVQ_STATS_CONFIG_GENERATOR
from sandbox.projects.advq.autodeploy.common import MIN_ADVQ_DEPLOY_BINARY, BRANCH_RELEASE, BRANCH_TESTING, \
    DEFAULT_PROCESS_TIMEOUT, _build_config_res
from sandbox.projects.advq.common import validate_arcadia_rev
from sandbox.projects.advq.common.configs import AdvqDeployConfiguration
from sandbox.projects.advq.common.parameters import CommonPhitsParameters, releaseTo_selector, SandboxParameters
from sandbox.projects.advq.common.sandbox_utils import get_sandbox_env_from_parameters
from sandbox.sdk2 import ResourceData, Resource
from sandbox.sdk2.helpers import subprocess as sp

RAW_PLATFORM_HOSTNAMES_REV = 3301574
DEPLOYMENT_CONFIG_REV = 5461375

SANDBOX_IDS_REV = 5786393


class AdvqAutodeployHelperGenTestingConfig(sdk2.Task):
    """
    Generate new configs with testing branch.

    We generate one coordinated config set, and then create separate resource for each config.
    Output parameter result_config_queue is a JSON-encoded (i.e. string) list of pairs (service_id, int(res)).
    """

    class Parameters(sdk2.Task.Parameters):
        input_service_info_res = sdk2.parameters.Resource("Service info resource")
        input_previous_configs_res = sdk2.parameters.Resource("Services' previous configs resource")

        phits_parameters = CommonPhitsParameters
        input_data_release = releaseTo_selector("Databases release status", with_empty=True)

        with_sumhits = sdk2.parameters.Bool("With sumhits")
        with_sumhits_mini = sdk2.parameters.Bool("With sumhits mini")
        with_weekly = sdk2.parameters.Bool("With weeklyhits")
        with_monthly = sdk2.parameters.Bool("With monthlyhits")

        with with_weekly.value[True]:
            lock_weekly = sdk2.parameters.Bool("Lock weekly databases", required=False, default=False)

        flat_layout = sdk2.parameters.Bool(
            "Flat layout",
            required=True,
            default=False,
            description="With flat layout, each machine has full set of chunks (e.g. for spikes)"
        )

        new_only = sdk2.parameters.Bool("Release config only if it differs from old one")

        replication_factor = sdk2.parameters.Integer("Replication factor", required=True, default=1,
                                                     description="Replicate files over given number of hosts")
        clear_tags = sdk2.parameters.Bool("Clear files tags", required=True, default=False,
                                          description="Remove from tags everything, that is not a valid host name")

        raw_platform_hostnames = sdk2.parameters.Bool("Raw platform hostnames")

        deploy_config_res = sdk2.parameters.Resource(
            "Deploy configuration resource JSON",
            resource_type=AdvqDeployConfiguration,
        )
        layout_memory_limit = sdk2.parameters.Float("Memory limit for generated config", required=False)
        layout_disk_limit = sdk2.parameters.Float("Disk limit for generated config", required=False)

        ram_disk_limit = sdk2.parameters.Bool("required disk_free_space > ram_free_space", required=True, default=False)

        databases = sdk2.parameters.List(
            "Databases (default list if empty)",
            description="""In any incomprehensible situation, use default list.

        Default list depends on phits type and is a list of keys of
        advq.generation.common.config.StandardConfig.DATABASES[phits_type]""")

        nanny_services = sdk2.parameters.List("Nanny services to deploy to")
        platform_services = sdk2.parameters.List(
            "Platform components to deploy to (<project>.<application>.<environment>.<component>)")

        force_deploy = sdk2.parameters.Bool("Force deploy", description="""Perform deploy even if service is
        initially in inconsistent state.
        """)
        redeploy = sdk2.parameters.Bool("Deploy from scratcj", description="""Deploy from zero config""", default=False)
        with_http = sdk2.parameters.Bool("With HTTP links (fresh backend required)")
        with_sandbox_ids = sdk2.parameters.Bool("Add Sandbox IDs for useful info in logs (fresh backend required)")

        sandbox_parameters = SandboxParameters

        with sdk2.parameters.Output:
            result_config_queue_json = sdk2.parameters.String("JSON-encoded deploy queue")
            result_report_json = sdk2.parameters.String("JSON-encoded report")
            result_new_config = sdk2.parameters.Bool("Resulting config is different from previous one")

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        validate_arcadia_rev(
            self.Parameters.phits_parameters.advq_build_binaries,
            [RAW_PLATFORM_HOSTNAMES_REV, MIN_ADVQ_DEPLOY_BINARY],
        )

        binaries = ResourceData(self.Parameters.phits_parameters.advq_build_binaries)

        if (not self.Parameters.nanny_services) and (not self.Parameters.platform_services):
            raise TaskFailure("Please, define at least Nanny service or Platform component to deploy to")

        config_generator_path = str(binaries.path.joinpath(ADVQ_STATS_CONFIG_GENERATOR))

        self._generate_configs(config_generator_path)

    #
    # Phase 1. Generation of test configs.
    #
    def _generate_configs(self, config_generator_path):
        """
        Generate config after self._get_service_info.

        Configs dict JSON is stored to resource with ID self.Context.generated_configs_res_id.

        :param config_generator_path: path to advq_stats_config_generator
        :return: None
        """
        # Load from previous resources
        with ResourceData(self.Parameters.input_previous_configs_res).path.open('rb') as inp:
            previous_configs_per_service = json.load(inp)
        with ResourceData(self.Parameters.input_service_info_res).path.open('rb') as inp:
            service_info = json.load(inp)

        for service_id in self.Parameters.nanny_services + self.Parameters.platform_services:
            missing_config = False
            if service_id in previous_configs_per_service and not self.Parameters.redeploy:
                res_id_or_undefined = previous_configs_per_service[service_id]
                if isinstance(res_id_or_undefined, six.integer_types):
                    config_res_data = ResourceData(Resource[res_id_or_undefined])
                    with config_res_data.path.joinpath(CONFIG_FILENAME).open(mode='rb') as config_file:
                        config = json.load(config_file)
                    service_info[service_id]['last_config'] = config
                else:
                    missing_config = True
            else:
                missing_config = True

            if missing_config:
                message = "No valid old config info for {!r}".format(service_id)
                if self.Parameters.force_deploy:
                    self.set_info("ATTENTION: {} (forced deploy)".format(message))
                else:
                    raise TaskFailure(message)
        service_info_input_path = tempfile.NamedTemporaryFile(prefix="gen_config_input").name
        report_path = tempfile.NamedTemporaryFile(prefix="report").name
        with open(service_info_input_path, 'wb') as service_info_out:
            json.dump(service_info, service_info_out)
        gen_testing_layouts_args = [
            'gen_layout',
            '--service-info-file', service_info_input_path,
            '--report-file', report_path,
        ]
        if self.Parameters.with_sumhits:
            gen_testing_layouts_args.append('--with-sumhits')
        if self.Parameters.with_sumhits_mini:
            gen_testing_layouts_args.append('--with-sumhits-mini')
        if self.Parameters.with_weekly:
            gen_testing_layouts_args.append('--with-weekly')
        if self.Parameters.with_monthly:
            gen_testing_layouts_args.append('--with-monthly')
        if self.Parameters.lock_weekly:
            gen_testing_layouts_args.append('--lock-weekly')
        if self.Parameters.flat_layout:
            gen_testing_layouts_args.append('--flat-layout')
        if self.Parameters.input_data_release != 'any':
            gen_testing_layouts_args.extend(['--released', self.Parameters.input_data_release])
        if self.Parameters.new_only:
            gen_testing_layouts_args.append('--new-only')
        if self.Parameters.databases:
            gen_testing_layouts_args.extend(['--dbs', ','.join(self.Parameters.databases)])
        if self.Parameters.deploy_config_res:
            validate_arcadia_rev(
                self.Parameters.phits_parameters.advq_build_binaries,
                [DEPLOYMENT_CONFIG_REV],
            )
            gen_testing_layouts_args.extend([
                '--deploy-configuration-file',
                str(ResourceData(self.Parameters.deploy_config_res).path),
            ])
        if self.Parameters.layout_memory_limit:
            gen_testing_layouts_args.extend(['--memory-limit', str(self.Parameters.layout_memory_limit)])
        if self.Parameters.layout_disk_limit:
            gen_testing_layouts_args.extend(['--disk-limit', str(self.Parameters.layout_disk_limit)])
        if self.Parameters.raw_platform_hostnames:
            gen_testing_layouts_args.append('--use-raw-platform-hostnames')
        if self.Parameters.clear_tags:
            gen_testing_layouts_args.append('--clear-tags')
        if self.Parameters.ram_disk_limit:
            gen_testing_layouts_args.append('--ram-disk-common-limit')
        if self.Parameters.with_http:
            gen_testing_layouts_args.append('--with-http')
        if self.Parameters.with_sandbox_ids:
            validate_arcadia_rev(
                self.Parameters.phits_parameters.advq_build_binaries,
                [SANDBOX_IDS_REV],
            )
            gen_testing_layouts_args.append('--with-sandbox-ids')

        gen_testing_layouts_args.extend([
            '--keep-branch', BRANCH_RELEASE,
            '--create-branch', BRANCH_TESTING,
            '--phits-type', self.Parameters.phits_parameters.advq_phits_type,
            '--replication-factor', str(self.Parameters.replication_factor),
        ])
        new_config = True
        env = get_sandbox_env_from_parameters(self.Parameters.sandbox_parameters)

        with sdk2.helpers.ProcessLog(self,
                                     logger=logging.getLogger("config_generator_gen_layout"),
                                     stdout_level=logging.INFO,
                                     stderr_level=logging.ERROR) as pl:
            # сейчас бинарник а) не использует Sandbox OAuth токен, б) обращается
            # напрямую в большой Sandbox, т.к. конфиг с адресом тестового Sandbox ему не передаётся.
            # Нужно об этом не забывать при тестировании.
            try:
                configs = sp.check_output(
                    [config_generator_path] + gen_testing_layouts_args,
                    timeout=DEFAULT_PROCESS_TIMEOUT,
                    stderr=pl.stdout,
                    env=env
                )
            except sp.CalledProcessError as ex:
                if ex.returncode == 3:
                    raise TaskFailure("Generated layout exceeded memory or disk limitation, see logs")
                elif ex.returncode == 5:
                    new_config = False
                    configs = ex.output
                    self.set_info("Generated test layout is the same as the layout generated last time")
                else:
                    raise
        configs = json.loads(configs)
        # выставляем "result_new_config" в output этой задачи
        out_new_config = new_config
        result_queue = []
        for count, (service_id, config) in enumerate(six.iteritems(configs)):
            # если конфиг старый, то вместо _build_config_res использовать previous_configs_per_service[service_id]
            prev_config_res_id = previous_configs_per_service[service_id]
            if new_config or not isinstance(prev_config_res_id, six.integer_types):
                config_res_id = int(_build_config_res(self, service_id, config, "Testing", count=count))
                out_new_config = True
            else:
                config_res_id = prev_config_res_id
            result_queue.append((service_id, config_res_id))
        self.Parameters.result_new_config = out_new_config
        self.Parameters.result_config_queue_json = json.dumps(result_queue)
        with open(report_path, 'rb') as report_file:
            self.Parameters.result_report_json = report_file.read()
