# -*- coding: utf-8 -*-
import logging
import tempfile
import six

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.advq.artifacts import ADVQ_STATS_CONFIG_GENERATOR
from sandbox.projects.advq.autodeploy.common import MIN_ADVQ_DEPLOY_BINARY, DEFAULT_PROCESS_TIMEOUT, _get_configs, \
    _save_tmp_file
from sandbox.projects.advq.common.parameters import NannyTokenParameters, PlatformTokenParameters, CommonPhitsParameters, \
    releaseTo_selector, get_secret_from_vault
from sandbox.sdk2 import ResourceData, WaitTime
from sandbox.sdk2.helpers import subprocess as sp


class AdvqAutodeployHelperGetServiceInfo(sdk2.Task):
    """
    Helper task that collects information on current state of services.
    It generates two AdvqTempFile resources.
    """

    class Parameters(sdk2.Task.Parameters):
        nanny_token = NannyTokenParameters
        platform_token = PlatformTokenParameters
        phits_parameters = CommonPhitsParameters
        releaseTo = releaseTo_selector("Release branch for Platform service (if any)")
        force_deploy = sdk2.parameters.Bool("Force deploy", description="""Perform deploy even if service is
initially in inconsistent state.
""")

        raw_platform_hostnames = sdk2.parameters.Bool("Raw platform hostnames")

        service_wait = sdk2.parameters.Integer(
            "Wait time interval between retries for deploy, seconds (60-300sec for spikes, 1200 for normal, etc)",
            default=300)
        get_configs_retries = sdk2.parameters.Integer(
            "Get service retries number",
            default=10)
        nanny_services_with_port = sdk2.parameters.List("Nanny services to deploy to, with possible forced port")
        platform_services = sdk2.parameters.List(
            "Platform components to deploy to (<project>.<application>.<environment>.<component>)")
        max_offline_hosts = sdk2.parameters.Integer(
            "Max offline hosts",
            required=True,
            default=0,  # safe value for phits types without replication
            description="Do not fail if number of offline hosts is less than given value"
        )

        with sdk2.parameters.Output:
            # конфигурация сервиса service_id -> { "hosts" : ["hostname1:port1", "hostname2:port2", ...],
            #                                      "last_config": id конфига, который сейчас на хостах }
            result_service_info_res_id = sdk2.parameters.Integer("Service info resource ID")
            # конфиги, задеплоенные в сервисы: service_id -> sandbox_resource_id
            result_previous_configs_res_id = sdk2.parameters.Integer("Contents of services' previous configs")

    class Requirements(sdk2.Task.Requirements):
        cores = 1
        disk_space = 1024

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        binaries = ResourceData(self.Parameters.phits_parameters.advq_build_binaries)
        if self.Parameters.advq_build_binaries.arcadia_revision < MIN_ADVQ_DEPLOY_BINARY:
            raise TaskFailure("ADVQ_GENERATION_BINARIES has to be at least of r{}".format(MIN_ADVQ_DEPLOY_BINARY))

        if (not self.Parameters.nanny_services_with_port) and (not self.Parameters.platform_services):
            raise TaskFailure("Please, define at least Nanny service or Platform component to deploy to")

        config_generator_path = str(binaries.path.joinpath(ADVQ_STATS_CONFIG_GENERATOR))
        with self.memoize_stage.get_configs(commit_on_entrance=False, commit_on_wait=False):
            with self.memoize_stage.initialize_get_configs:
                self.Context.get_configs_retries_count = self.Parameters.get_configs_retries
            self._get_service_info(config_generator_path)

    #
    # Phase 0.  Getting services info (hosts, old configs)
    #
    def _get_service_info(self, config_generator_path):
        """
        Phase 0: get service info for each service, including previous config, if any.

        This method may resrtart with WaitTime.

        :param config_generator_path: path to advq_stats_config_generator
        :return: None
        """
        tokens_args = []
        nanny_token = get_secret_from_vault('nanny', self.Parameters.nanny_token)
        if nanny_token is not None:
            nanny_token_file = tempfile.NamedTemporaryFile(prefix='nanny_token_file_').name
            with open(nanny_token_file, 'wb') as out:
                out.write(nanny_token)
            tokens_args.extend(['--nanny-token-file', nanny_token_file])
        platform_token = get_secret_from_vault('platform', self.Parameters.platform_token)
        if platform_token is not None:
            platform_token_file = tempfile.NamedTemporaryFile(prefix='platform_token_file_').name
            with open(platform_token_file, 'wb') as out:
                out.write(platform_token)
            tokens_args.extend(['--platform-token-file', platform_token_file])

        service_info_args = ['gen_service_info', '--service-release', self.Parameters.releaseTo] + tokens_args
        if self.Parameters.nanny_services_with_port:
            service_info_args.append('--nanny')
            service_info_args.extend(self.Parameters.nanny_services_with_port)
        if self.Parameters.platform_services:
            service_info_args.append('--platform')
            service_info_args.extend(self.Parameters.platform_services)
            if platform_token is None:
                raise TaskFailure("Platform OAuth token is absolutely required for Platform deploy")
        if self.Parameters.raw_platform_hostnames:
            service_info_args.append('--use-raw-platform-hostnames')

        service_info_filename = 'service_info.json'
        with open(service_info_filename, 'wb') as service_info_file:
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("config_generator_gen_service_info")) as pl:
                sp.check_call(
                    [config_generator_path] + service_info_args,
                    timeout=DEFAULT_PROCESS_TIMEOUT,
                    stdout=service_info_file,
                    stderr=pl.stdout,
                )

        config_resources = _get_configs(self, config_generator_path, service_info_filename)
        old_config_resources_per_service = {}
        for service_id, hosts_to_cfg in six.iteritems(config_resources):
            valid_states = frozenset(
                res_or_error
                for res_or_error in six.itervalues(hosts_to_cfg)
                if isinstance(res_or_error, six.integer_types)  # i.e. not a error
            )
            if valid_states:
                latest_res = max(valid_states)
                failed_hosts = [host for host, res_or_error in six.iteritems(hosts_to_cfg) if res_or_error != latest_res]
            else:
                latest_res = None
                failed_hosts = list(six.iterkeys(hosts_to_cfg))

            old_config_resources_per_service[service_id] = latest_res
            if not self.Parameters.force_deploy:
                if len(failed_hosts) > self.Parameters.max_offline_hosts:
                    self.set_info("WARNING: too many hosts are not ready: {!r}".format(failed_hosts))
                    if self.Context.get_configs_retries_count <= 0:
                        # Достигли 0, не получив результат
                        raise TaskFailure("Failed to get consistent configs from some of service;"
                                          " global failure or stuck previous deploy?")
                    self.Context.get_configs_retries_count -= 1
                    self.set_info(
                        "Retrying {}/{}".format(
                            self.Context.get_configs_retries_count, self.Parameters.get_configs_retries
                        )
                    )
                    raise WaitTime(time_to_wait=self.Parameters.service_wait)
                elif len(hosts_to_cfg) == 0:
                    self.set_info(
                        "WARNING: {!r} has empty number of hosts.  Skipping".format(service_id))

        # Получили консистентный (по каждому ДЦ) набор конфигов; сохраняем в ресурс
        old_config_resources_per_service_path = "old_config_resources.json"
        tmp_old_config_per_service_res = _save_tmp_file(
            self,
            "Services' previous configs JSON",
            old_config_resources_per_service_path,
            json_data=old_config_resources_per_service,
        )
        self.Parameters.result_previous_configs_res_id = tmp_old_config_per_service_res.id
        tmp_service_info_res = _save_tmp_file(
            self,
            "Service info JSON",
            service_info_filename,
        )
        self.Parameters.result_service_info_res_id = tmp_service_info_res.id
        self.set_info("Service info res: {!r}".format(self.Context.service_info_res_id))
