# -*- coding: utf-8 -*-
import json
import logging
import os.path
import six
import shutil

from sandbox import sdk2
from sandbox.projects.advq.AdvqDeployConfigToHosts import CONFIG_DIR, CONFIG_FILENAME
from sandbox.projects.advq.common import AdvqTempFile
from sandbox.projects.advq.common.configs import (
    AdvqNormalPhitsConfigTesting, AdvqNormalPhitsConfigManYp, AdvqNormalPhitsConfigSas, AdvqSpikesPhitsConfig,
    AdvqNormalPhitsConfigVlaYp, AdvqVideoPhitsConfigMan, AdvqVideoPhitsConfigVla, AdvqVideoPhitsConfigTesting,
    AdvqForecastConfigSasYp, AdvqForecastConfigVlaYp, AdvqForecastConfigManYp
)
from sandbox.sdk2 import ResourceData
from sandbox.sdk2.helpers import subprocess as sp


# Строка, которая возвращается ADVQ HTTP API при вызовах типа /config/current, если версия в конфиге не определена
CONFIG_STATE_UNDEFINED = '[undefined]'
TMP_TTL = 7  # ttl для временных ресурсов; для разбора проблем -- больше длины обычных выходных

DEFAULT_PROCESS_TIMEOUT = 3 * 10 * 60  # обычный таймаут для запускаемых процессов, 30 минут
WAIT_TIMEOUT = 40 * 60  # таймаут ожидание лёгких тасков тасков, 40 минут
DATA_TEST_WAIT_TIMEOUT = 60 * 60  # 1 час, в секундах

MIN_ADVQ_DEPLOY_BINARY = 3621731  # ADVQ-1825, added --lock-weekly to gen_layout subcommand.

BRANCH_RELEASE = 'release'
BRANCH_TESTING = 'testing'

CONFIG_RESOURCE_TYPE_BY_SERVICE = {
    'advq_java_normal_sas': AdvqNormalPhitsConfigSas,
    'advq_java_normal_vla_yp': AdvqNormalPhitsConfigVlaYp,
    'advq_java_normal_man_yp': AdvqNormalPhitsConfigManYp,
    'advq_java_normal_testing': AdvqNormalPhitsConfigTesting,

    'advq.java-spikes.prod-platform.spikes': AdvqSpikesPhitsConfig,  # production, stopped
    'advq.java-spikes.test.spikes': AdvqSpikesPhitsConfig,  # testing, stopped
    'advq.java-video.production.backend-man': AdvqVideoPhitsConfigMan,  # production
    'advq.java-video.production.backend-vla': AdvqVideoPhitsConfigVla,  # production
    'advq.java-video.testing.backend-man': AdvqVideoPhitsConfigTesting,  # testing

    'advq_java_forecast_sas_yp': AdvqForecastConfigSasYp,
    'advq_java_forecast_vla_yp': AdvqForecastConfigVlaYp,
    'advq_java_forecast_man_yp': AdvqForecastConfigManYp,
}


def list_as_tuples_hook(the_dict):
    """object_hook для json.load, который списки (JSON-массивы) конвертирует в tuple.

    Этот хук не сконвертирует [1,2] в структуре [0, [1, 2], 3], но у нас такие не встречаются.
    """
    for key, val in list(six.iteritems(the_dict)):
        if isinstance(val, list):
            the_dict[key] = tuple(val)
    return the_dict


def _get_configs(task, config_generator_path, service_info_filename):
    """
    Helper function to get from hosts resource IDs of configs.

    :param task: Task object that calls this function.
    :param config_generator_path: config generator binary path
    :param service_info_filename: service info JSON file name
    :return: Contents of get_configs, JSON-decoded.
    :rtype Dict(str, Dict(str, Or(List, int)))
    """
    # базовые конфиги, от которых делаем новые
    get_configs_args = ['get_configs',
                        '--service-release', task.Parameters.releaseTo,
                        '--service-info-file', service_info_filename]

    with sdk2.helpers.ProcessLog(task, logger=logging.getLogger("config_generator_gen_service_info")) as pl:
        config_resources_json = sp.check_output(
            [config_generator_path] + get_configs_args,
            timeout=DEFAULT_PROCESS_TIMEOUT,
            stderr=pl.stdout,
        )
    config_resources = json.loads(config_resources_json, object_hook=list_as_tuples_hook)
    # DEBUG
    task.set_info(__import__('pprint').pformat(config_resources), do_escape=True)
    return config_resources


def _save_tmp_file(task, description, path, json_data=None):
    """
    Create resource from local path.

    :param task: Task that will own new resources
    :param description: Textual resource description
    :param path: Path to file
    :param json_data: if not None, json.dump to path
    :return: AdvqTempFile resource, in READY state
    """
    res = AdvqTempFile(task, description, os.path.basename(path), ttl=TMP_TTL)

    rd = ResourceData(res)
    result_path = rd.path
    if json_data is not None:
        with result_path.open('wb') as out:
            json.dump(json_data, out)
    else:
        if os.path.abspath(path) != os.path.abspath(str(result_path)):
            shutil.copy(path, str(result_path))
    rd.ready()
    return res


def _build_config_res(task, service_id, config_dict, config_description, count=None):
    """
    Build config resource with proper paths inside.

    :param task: task that resource will belong to
    :param service_id: service ID (Nanny service ID or Platform component name)
    :param config_dict: Config as python dict
    :param config_description: Description string ("testing" or "release") for resource title
    :type config_description str
    :return:
    """
    assert isinstance(config_dict, dict)
    res_class = CONFIG_RESOURCE_TYPE_BY_SERVICE[service_id]
    if count is None:
        config_dir = CONFIG_DIR
    else:
        config_dir = CONFIG_DIR + '{:04d}'.format(count)
    res = res_class(
        task, "{} config for {}".format(config_description.capitalize(), service_id),
        config_dir,
        # TODO(monoid) все эти параметры пока не прокидываются из генератора конфигов.
        advq_start_date="",
        advq_end_date="",
        advq_period_size=0,
        advq_resource_count=0,
        advq_dbs='',
        advq_phits_type=task.Parameters.phits_parameters.advq_phits_type,
        advq_resources='[]',
    )
    rd = ResourceData(res)
    rd.path.mkdir(0o755, parents=True, exist_ok=True)
    with rd.path.joinpath(CONFIG_FILENAME).open('wb') as out_file:
        json.dump(config_dict, out_file, sort_keys=True)
    rd.ready()
    return res
