# -*- coding: utf-8 -*-
import logging
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.projects.resource_types import advq_releasers
from sandbox.projects.resource_types.releasers import direct_sre
from sandbox.sdk2.resource import Attributes


BROADMATCH_NORM_DICT_FILENAME = 'norm_dict.latest.trie'
BROADMATCH_RENORM_DICT_FILENAME = 'renorm_dict.latest.trie'


MRKIT_WITH_DEBUG_REV = 3266061

# огранчения на ревизию бинарников для задач, которые используют бинарники с YT.
YT_MINIMAL_REV = 3557158  # ADVQ-1808

# общее ограничение для генерации phits и для антифрода, чтобы они не разъезжались
PHITS_TABLES_MINIMAL_REV = 4280758  # ADVQ-1916

SHELL_COMMAND_PREFIX = ['/bin/bash', '-o', 'pipefail', '-e', '-c']

DEFAULT_PROCESS_TIMEOUT = 3 * 10 * 60  # обычный таймаут для запускаемых процессов, 30 минут


def validate_arcadia_rev(binary_resource, min_revisions):
    assert isinstance(binary_resource, (AdvqGenerationBinaries, OfflinePhitsArtifacts, OfflinePkzBinaries))
    logging.debug("list={!r}, tuple={!r}".format(list, tuple))
    assert isinstance(min_revisions, (list, tuple))
    binary_revision = binary_resource.arcadia_revision
    if min_revisions:
        required_revision = max(min_revisions)
        if binary_revision < required_revision:
            raise TaskFailure(
                "Generation binaries from revision {} required, but {} provided".format(
                    required_revision, binary_revision
                ))


class AdvqTempFile(sdk2.Resource):
    """
    Временные файлы, которые сохраняются как ресурсы между перезапусками WaitTask/WaitTime.
    """
    releasable = False
    any_arch = True
    auto_backup = False
    ttl = 2


class AdvqBroadmatchDatabases(sdk2.Resource):
    """
    Два файла -- norm_dict.latest.trie, renorm_dict.latest.trie
    Содержат trie для лемматизации слов, используется для генерации sumhits (а именно, SimilarityHexdigest).

    Дата генерации содержится в атрибутах, а не в имени файлов.
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True

    generation_date = Attributes.String("Generation date")


class AdvqGenerationBinaries(sdk2.Resource):
    """
    Утилиты для генерации таблиц и индексов из разных источников:
    1. hitlog_generator
    2. phits_generator
    3. phits_db_generator
    4. chrono_db_generator + mrkit_legacy_reader
    5. advq_phits_antifraud

    TODO(monoid) создать для удобства отдельный таск (YaMake, sdk1) со всеми путями, заданными по умолчанию.
    """
    releasable = True
    executable = True
    releasers = direct_sre + advq_releasers


class OfflinePhitsArtifacts(sdk2.Resource):
    """
    Артефакты сборки offline-phits
    1. direct_export - расчёт рекламных фраз директа
    2. offline_phits_util.jar - java-parser
    """
    releasable = True
    executable = True
    releasers = direct_sre + advq_releasers


class OfflinePkzBinaries(sdk2.Resource):
    """
    Артефакты сборки offline pkz.
    """
    releasable = True
    executable = True
    releasers = direct_sre + advq_releasers


class PokazometerGenerationBinaries(sdk2.Resource):
    """
    Артефакты сборки генерации pokazometer (таблицы, базы).
    """
    releasable = True
    executable = True
    releasers = direct_sre + advq_releasers


class AdvqPhitsDatabaseChunk(sdk2.Resource):
    """
    Один чанк базы phits.  В атрибутах ресурса содержится:

    1. строка advq_phits_type (normal/partial/video/spikes/etc)
    2. строка advq_db (rus/tur/rus-robots/tur-robots etc)
    3. строка advq_epodate (дата и, возможно, эпоха: {YYYY}{MM}{dd} или {YYYY}{MM}{dd}-{epoch};
       у дельт это поле имеет вид {YYYY}{MM}{dd}-delta{epoch})
    4. отдельно advq_date без эпохи
    5. отдельно целочисленная эпоха
    6. целое advq_total_chunks (общее количество чанков)
    7. целое advq_chunk (номер чанка от 1 до total_chunks)
    8. булево advq_is_delta (признак, что ресурс является дельтой, которую нужно использовать с базовым phits).

    Кроме того, всё это повторяется в имени файла:
    phits_{phits_type}_{db}_{epodate}.{chunk_number}.{total_chunks}.db

    (признак is_delta содержится в epodate)
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 730

    advq_phits_type = Attributes.String("phits_type (normal/partial/video/etc)", required=True)
    advq_db = Attributes.String("phits db name", required=True)
    advq_date = Attributes.String("phits date without epoch", required=True)
    advq_epoch = Attributes.Integer("phits epoch", required=True)
    advq_epodate = Attributes.String("phits date with epoch", required=True)
    advq_total_chunks = Attributes.Integer("total chunk number", required=True)
    advq_chunk = Attributes.Integer("number of this particular chunk, starting from 1", required=True)
    advq_is_delta = Attributes.Bool("Is delta (not base+delta combined)", required=True)
    binaries_arcadia_revision = Attributes.Integer("Generation utils' Arcadia revision")

    def __init__(self, *args, **kwargs):
        if self.__class__ == AdvqPhitsDatabaseChunk:
            raise ValueError("Only subclasses of {} can be instantiated".format(AdvqPhitsDatabaseChunk))
        super(AdvqPhitsDatabaseChunk, self).__init__(*args, **kwargs)


# Важно, чтобы имя ресурса формировалось по шаблону
# 'ADVQ_{}_PHITS_DATABASE_CHUNK'.format(str(phits_type).upper())
class AdvqNormalPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    pass


class AdvqHurmalPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    pass


class AdvqVideoPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    pass


class AdvqPartialPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    pass


class AdvqSpikesPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    pass


class AdvqExperimentPhitsDatabaseChunk(AdvqPhitsDatabaseChunk):
    """Experiments etc"""
    pass


class AdvqSumhitsDatabaseChunk(sdk2.Resource):
    """
    Один чанк базы sumhits.  В атрибутах ресурса содержится:

    1. строка advq_phits_type (normal/partial/video/spikes/etc)
    2. строка advq_db (rus/tur/rus-robots/tur-robots etc)
    3. строка advq_epodate (дата и, возможно, эпоха: {YYYY}{MM}{dd} или {YYYY}{MM}{dd}-{epoch};
       у дельт это поле имеет вид {YYYY}{MM}{dd}-delta{epoch})
    4. строка advq_start_date (начальная дата, включительно)
    5. строка advq_last_date (конечная дата, включительно)
    6. строка period_size (количество периодов -- дней, часов, получасовых, в зависимости от advq_phits_type)
    7. целое advq_total_chunks (общее количество чанков)
    8. целое advq_chunk (номер чанка от 1 до total_chunks)

    Кроме того, всё это повторяется в имени файла:
    sumhits_{phits_type}_{db}_{epodate}.{chunk_number}.{total_chunks}.db
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 60

    advq_phits_type = Attributes.String("phits_type (normal/partial/video/etc)", required=True)
    advq_db = Attributes.String("phits db name", required=True)
    advq_date = Attributes.String("date without epoch", required=True)
    advq_epoch = Attributes.Integer("epoch", required=True)
    advq_epodate = Attributes.String("date with epoch", required=True)
    advq_start_date = Attributes.String("start date", required=True)
    advq_last_date = Attributes.String("last date", required=True)
    period_size = Attributes.Integer("Period count", required=True)
    advq_total_chunks = Attributes.Integer("total chunk number", required=True)
    advq_chunk = Attributes.Integer("number of this particular chunk, starting from 1", required=True)
    advq_is_delta = Attributes.Bool("Is delta (not base+delta combined)", required=True)
    binaries_arcadia_revision = Attributes.Integer("Generation utils' Arcadia revision")

    def __init__(self, *args, **kwargs):
        if self.__class__ == AdvqSumhitsDatabaseChunk:
            raise ValueError("Only subclasses of {} can be instantiated".format(AdvqSumhitsDatabaseChunk))
        super(AdvqSumhitsDatabaseChunk, self).__init__(*args, **kwargs)


# Важно, чтобы имя ресурса формировалось по шаблону
# 'ADVQ_{}_PHITS_DATABASE_CHUNK'.format(str(phits_type).upper())
class AdvqNormalSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqMiniSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqHurmalSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqVideoSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqPartialSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqSpikesSumhitsDatabaseChunk(AdvqSumhitsDatabaseChunk):
    pass


class AdvqChronoDatabaseChunk(sdk2.Resource):
    """
    Один чанк базы chrono.  В атрибутах ресурса содержится:

    1. строка advq_phits_type (normal/partial/video)
    2. строка advq_db (rus/tur/rus-robots/tur-robots etc.)
    3. строка advq_epodate (дата и, возможно, эпоха; у дельт это поле имеет вид {YYYY}{MM}{dd}-{epoch}.delta)
    4. целое advq_total_chunks (общее количество чанков)
    5. целое advq_chunk (номер чанка от 1 до total_chunks)
    6. строка advq_chrono_type ('weekly' или 'monthly')

    Кроме того, всё это повторяется в имени файла:
    chrono_{advq_phits_type}_{db}_{advq_chrono_type}hits_{epodate}_{chunk_number}_{total_chunks}_{task_id}.db
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 'inf'

    advq_phits_type = Attributes.String("phits_type (normal/partial/video/etc)", required=True)
    advq_db = Attributes.String("phits db name", required=True)
    advq_date = Attributes.String("phits date without epoch", required=True)
    advq_epoch = Attributes.Integer("phits epoch", required=True)
    advq_epodate = Attributes.String("phits date with epoch", required=True)
    advq_total_chunks = Attributes.Integer("database total chunk number", required=True)
    advq_chunk = Attributes.Integer("number of this particular chunk, starting from 1", required=True)
    advq_is_delta = Attributes.Bool("Is delta (not base+delta combined)", required=True)
    advq_chrono_type = Attributes.String("'weekly' or 'monthly'", required=True)
    advq_input_table = Attributes.String("Input table")
    arcadia_revision = Attributes.Integer("Generation utils Arcadia revision")


class AdvqNormalChronoDatabaseChunk(AdvqChronoDatabaseChunk):
    pass


class AdvqHurmalChronoDatabaseChunk(AdvqChronoDatabaseChunk):
    pass


class AdvqLegacyNormalChronoDatabaseChunk(AdvqNormalChronoDatabaseChunk):
    """Same as AdvqNormalChronoDatabaseChunk, but generated always as a single
    volume and shipped to ADVQ7."""


PHITS_RESOURCE_CLASS_BY_TYPE = {
    'normal': AdvqNormalPhitsDatabaseChunk,
    'hurmal': AdvqHurmalPhitsDatabaseChunk,
    'video': AdvqVideoPhitsDatabaseChunk,
    'partial': AdvqPartialPhitsDatabaseChunk,
    'spikes': AdvqSpikesPhitsDatabaseChunk,
    'experiment': AdvqExperimentPhitsDatabaseChunk,
}

SUMHITS_RESOURCE_CLASS_BY_TYPE = {
    'normal': AdvqNormalSumhitsDatabaseChunk,
    'mini': AdvqMiniSumhitsDatabaseChunk,
    'hurmal': AdvqHurmalSumhitsDatabaseChunk,
    'video': AdvqVideoSumhitsDatabaseChunk,
    'partial': AdvqPartialSumhitsDatabaseChunk,
    'spikes': AdvqSpikesSumhitsDatabaseChunk,
}

CHRONO_RESOURCE_CLASS_BY_CHUNK = {
    'normal': AdvqNormalChronoDatabaseChunk,
    'hurmal': AdvqHurmalChronoDatabaseChunk,
    # для других пока нет надобности
}


def get_phits_resource_class(phits_type):
    return PHITS_RESOURCE_CLASS_BY_TYPE[str(phits_type)]


def get_sumhits_resource_class(phits_type):
    return SUMHITS_RESOURCE_CLASS_BY_TYPE[str(phits_type)]


def get_chrono_resource_class(phits_type):
    return CHRONO_RESOURCE_CLASS_BY_CHUNK[str(phits_type)]


class PkzDatabaseChunk(sdk2.Resource):
    """
    Один чанк базы прогнозатора. В атрибутах ресурса содержится:

    1. строка pkz_db (rus/tur etc.)
    2. строка pkz_date (дата)
    3. целое pkz_total_chunks (общее количество чанков)
    4. целое pkz_chunk (номер чанка от 1 до total_chunks)
    5. строка pkz_input_table (ытевая таблица)

    Кроме того, всё это повторяется в имени файла:
    pkz_{db}_{chunk_number}_{total_chunks}.db
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 'inf'

    pkz_db = Attributes.String("pkz db name", required=True)
    pkz_date = Attributes.String("date", required=True)
    pkz_total_chunks = Attributes.Integer("database total chunk number", required=True)
    pkz_chunk = Attributes.Integer("number of this particular chunk, starting from 1", required=True)
    pkz_input_table = Attributes.String("Input table")
    arcadia_revision = Attributes.Integer("Generation utils Arcadia revision")


def date_epoch_delta_to_epodate(date, epoch, is_delta):
    if epoch == 0:
        assert not is_delta, "Deltas cannot have zero epoch: {!r}, {!r}, {!r}".format(date, epoch, is_delta)
        return date
    else:
        if is_delta:
            delta = 'delta'
        else:
            delta = ''
        return '{}-{}{}'.format(date, delta, epoch)


class AdvqApiDataTestReport(sdk2.Resource):
    releasable = False
    any_arch = True

    released_meta = sdk2.Attributes.String("Released API meta", required=True)
    testing_meta = sdk2.Attributes.String("Testing API meta", required=True)

    test_success = sdk2.Attributes.Bool("Testing successful?", required=True)
    tested_config = sdk2.Attributes.String("Tested configuration", required=True)
    startrek_ticket_id = sdk2.Attributes.String("Startrek ticket id for test failures", required=False)
