# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox.projects.resource_types import advq_releasers
from sandbox.projects.resource_types.releasers import direct_sre
from sandbox.sdk2 import Attributes

# actually, data/layout.conf, but only at final stage.
ADVQ_CONFIG_NAME = 'layout.conf'
ADVQ_DEPLOY_CONFIGURATION = 'advq_deploy_config.json'


class AdvqDeployConfiguration(sdk2.Resource):
    """
    Static config for ADVQ deploy.  Used only by autodeploy tasks.  More flexible
    than Sandbox task parameters.

    Contents: JSON file named "advq_deploy_config.json".  JSON consists of dict
    where keys are DC designators (Nanny service name or Platform component,
    like "advq_java_normal_man"), and values currently are dicts with two fields:
    "layout_memory_limit" and "layout_disk_limit" (integers in GB, can be absent
    as "no limit").

    Example:

    {"advq_java_normal_man": {"layout_memory_limit": 450}}
    """
    auto_backup = True
    ttl = 'inf'


class AdvqPhitsConfig(sdk2.Resource):
    """
    Group of phits databases.
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 14

    # Параметры для деплоя в Platform
    platform_stable_environments = ()
    platform_testing_environments = ()

    advq_start_date = Attributes.String("Start (min) date for group", required=True)
    advq_end_date = Attributes.String("End (max) date for group, inclusive", required=True)
    advq_period_size = Attributes.Integer("Number of consecutive periods", required=True)
    advq_resource_count = Attributes.Integer(
        "Number of phits resources, should be equal to chunk count, including deltas", required=True)
    advq_dbs = Attributes.String("Comma-separted sorted list of dbs", required=True)
    advq_phits_type = Attributes.String("Phits type", required=True)
    advq_resources = Attributes.String("JSON-encoded list of databases' resource IDs")
    # TODO добавить вызовы lock_memory (опцию в конфиге)

    def __init__(self, *args, **kwargs):
        if self.__class__ == AdvqPhitsConfig:
            raise ValueError("Only subclasses of {} can be instantiated".format(AdvqPhitsConfig))
        super(AdvqPhitsConfig, self).__init__(*args, **kwargs)


class AdvqNormalPhitsConfig(AdvqPhitsConfig):
    def __init__(self, *args, **kwargs):
        if self.__class__ == AdvqNormalPhitsConfig:
            raise ValueError("Only subclasses of {} can be instantiated".format(AdvqNormalPhitsConfig))
        super(AdvqNormalPhitsConfig, self).__init__(*args, **kwargs)


class AdvqNormalPhitsConfigSas(AdvqNormalPhitsConfig):
    """
    Normal phits config for SAS.
    """
    pass


class AdvqNormalPhitsConfigVlaYp(AdvqNormalPhitsConfig):
    pass


class AdvqNormalPhitsConfigManYp(AdvqNormalPhitsConfig):
    pass


class AdvqNormalPhitsConfigTesting(AdvqNormalPhitsConfig):
    pass


class AdvqVideoPhitsConfig(AdvqPhitsConfig):
    pass


class AdvqPartialPhitsConfig(AdvqPhitsConfig):
    pass


class AdvqSpikesPhitsConfig(AdvqPhitsConfig):
    platform_stable_environments = ("advq.java-spikes.prod-platform",)
    platform_testing_environments = ("advq.java-spikes.test",)


class AdvqVideoPhitsConfigMan(AdvqVideoPhitsConfig):
    platform_stable_environments = ("advq.java-video.production",)


class AdvqVideoPhitsConfigVla(AdvqVideoPhitsConfig):
    platform_stable_environments = ("advq.java-video.production",)


class AdvqVideoPhitsConfigTesting(AdvqVideoPhitsConfig):
    """ Testing environment for ADVQ, video-based """
    platform_stable_environments = ("advq.java-video.testing",)


# Код ниже (PHITS_CONFIG_CLASS_BY_PHITS_TYPE, get_phits_config_class_by_type) используется
# для AdvqGroupSpikesDatabases, который с запуском нормального автодеплоя стал не нужен.
PHITS_CONFIG_CLASS_BY_PHITS_TYPE = {
    'video': AdvqVideoPhitsConfig,
    'partial': AdvqPartialPhitsConfig,
    'spikes': AdvqSpikesPhitsConfig,
    ('normal', 'sas'): AdvqNormalPhitsConfigSas,
    ('normal', 'vla_yp'): AdvqNormalPhitsConfigVlaYp,
    ('normal', 'man_yp'): AdvqNormalPhitsConfigManYp,
}


def get_phits_config_class_by_type(phits_type, data_center=None):
    if data_center is None:
        return PHITS_CONFIG_CLASS_BY_PHITS_TYPE[str(phits_type)]
    else:
        return PHITS_CONFIG_CLASS_BY_PHITS_TYPE[(str(phits_type), str(data_center))]


class AdvqForecastConfig(sdk2.Resource):
    """
    Group of pkz databases.
    """
    releasable = True
    releasers = direct_sre + advq_releasers
    any_arch = True
    auto_backup = True
    ttl = 14

    forecast_start_date = Attributes.String("Start (min) date for group", required=True)
    forecast_end_date = Attributes.String("End (max) date for group, inclusive", required=True)
    forecast_period_size = Attributes.Integer("Number of consecutive periods", required=True)
    forecast_resource_count = Attributes.Integer(
        "Number of pkz resources, should be equal to chunk count, including deltas", required=True)
    forecast_dbs = Attributes.String("Comma-separted sorted list of dbs", required=True)
    forecast_resources = Attributes.String("JSON-encoded list of databases' resource IDs")
    # TODO добавить вызовы lock_memory (опцию в конфиге)

    def __init__(self, *args, **kwargs):
        if self.__class__ == AdvqForecastConfig:
            raise ValueError("Only subclasses of {} can be instantiated".format(AdvqForecastConfig))
        super(AdvqForecastConfig, self).__init__(*args, **kwargs)


class AdvqForecastConfigSasYp(AdvqForecastConfig):
    pass


class AdvqForecastConfigVlaYp(AdvqForecastConfig):
    pass


class AdvqForecastConfigManYp(AdvqForecastConfig):
    pass
