# -*- coding: utf-8 -*-
from sandbox import sdk2
import sandbox.common.utils
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import sandboxapi
from sandbox.projects.advq.common import AdvqGenerationBinaries


class AdvqBuildBinariesResourceParameter(sdk2.parameters.Resource):
    resource_type = AdvqGenerationBinaries
    resource_released = ctt.ReleaseStatus.STABLE
    required = False

    @sandbox.common.utils.classproperty
    def default_value(cls):
        resources = sandboxapi.Sandbox().list_resources(
            resource_type=cls.resource_type,
            omit_failed=True,
            all_attrs={'released': cls.resource_released},
            limit=1,
        )
        if resources:
            return resources[0].id
        else:
            return None


class YtParameters(sdk2.Task.Parameters):
    yt_proxy = sdk2.parameters.String("YT server name", required=True)
    yt_token_vault_user = sdk2.parameters.String("YT key vault owner")
    yt_token_vault_name = sdk2.parameters.String("YT key vault name")


class NannyTokenParameters(sdk2.Task.Parameters):
    nanny_token_vault_user = sdk2.parameters.String("Nanny token vault owner")
    nanny_token_vault_name = sdk2.parameters.String("Nanny token vault name")


class PlatformTokenParameters(sdk2.Task.Parameters):
    platform_token_vault_user = sdk2.parameters.String("Platform token vault owner")
    platform_token_vault_name = sdk2.parameters.String("Platform token vault name")


class SandboxParameters(sdk2.Task.Parameters):
    sandbox_token_vault_user = sdk2.parameters.String("Sandbox token vault owner")
    sandbox_token_vault_name = sdk2.parameters.String("Sandbox token vault name")


class StartrekParameters(sdk2.Task.Parameters):
    startrek_token_vault_user = sdk2.parameters.String("Startrek token vault owner")
    startrek_token_vault_name = sdk2.parameters.String("Startrek token vault name")


def phits_type_radio_group(*args, **kwargs):
    with sdk2.parameters.RadioGroup(*args, **kwargs) as setter:
        setter.values["normal"] = setter.Value(value="normal")
        setter.values["mini"] = setter.Value(value="mini")
        setter.values["video"] = setter.Value(value="video")
        setter.values["partial"] = setter.Value(value="partial")
        setter.values["spikes"] = setter.Value(value="spikes")
        setter.values["hurmal"] = setter.Value(value="hurmal")
    return setter


class PhitsParameters(YtParameters):
    """
    Parameters common for many phits-building tasks.

    Legacy, use CommonPhitsParameters.
    """
    advq_phits_type = phits_type_radio_group("Source data type", required=True)
    advq_build_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation utils resource", required=True)


class CommonPhitsParameters(sdk2.Task.Parameters):
    """
    Parameters for processing phits and derived bases/tables (sumhits, chrono).
    """
    advq_phits_type = phits_type_radio_group("Source data type", required=True)
    advq_build_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation utils resource")


def get_secret_from_vault(prefix, vault_params):
    vault_name = getattr(vault_params, prefix + '_token_vault_name')
    if vault_name:
        vault_user = getattr(vault_params, prefix + '_token_vault_user')
        if vault_user:
            token = sdk2.Vault.data(vault_user, vault_name)
        else:
            token = sdk2.Vault.data(vault_name)
        return token
    else:
        return None


def log_verbosity_radio_group(levels):
    """
    Generate radio buttons for advq/generation/common/util/log.py helper

    levels - list of increasing levels (e.g. ["INFO", "DEBUG"])
    """
    assert levels

    with sdk2.parameters.RadioGroup("Increase logging level") as rg:
        rg.values["default"] = rg.Value("default", default=True)
        for i, level in enumerate(levels, start=1):
            rg.values['-%s' % ('v' * i,)] = rg.Value(value=level)

    return rg


def get_log_verbosity_flag(radio_group):
    """
    Return flag value generated by log_verbosity_radio_group or None if
    level shouldn't be changed
    """
    if str(radio_group) == "default":
        return None
    else:
        return str(radio_group)


def releaseTo_selector(title, with_empty=False):
    """Radio button for releasing branch selection"""
    with sdk2.parameters.RadioGroup(title) as releaseTo:
        if with_empty:
            assert 'any' not in sandboxapi.RELEASE_STATUSES
            releaseTo.values['any'] = releaseTo.Value(value='any', default=True)
        for val in sandboxapi.RELEASE_STATUSES:
            releaseTo.values[val] = releaseTo.Value(value=val)
    return releaseTo


def releaseTo_params():
    """
    Usage:

    class MyTask(sdk2.Task):
        class Parameters(sdk2.Task.Parameters):
            release_new_resource, releaseTo = releaseTo_params()
    """
    release_new_resource = sdk2.parameters.Bool(
        "Release new resource when it is created.",
        default=False,
        required=True,
    )

    with release_new_resource.value[True]:
        releaseTo = releaseTo_selector(title="Automatically release resource to", with_empty=False)

    return release_new_resource, releaseTo


def convert_ttl(ttl_param):
    """
    Convert user input to ttl.

    >>> convert_ttl(0)
    'inf'
    >>> convert_ttl(1)
    1
    >>> convert_ttl(5)
    5

    :param ttl_param: ttl from user input
    :return: 'inf' if ttl is 0, else ttl itself.
    """
    ttl_param = int(ttl_param)  # иногда приходит float
    return 'inf' if ttl_param == 0 else ttl_param
