# -*- coding: utf-8 -*-
import logging
import posixpath
import random
import string
from collections import defaultdict

import operator
import six
import six.moves

TYPE = 'type'
SANDBOX_CREATION_TIME = 'sandbox_creation_time'
SANDBOX_RESOURCE_ID = 'sandbox_resource_id'
RESOURCE_RELEASE_TYPE = 'stable'

logger = logging.getLogger('upload_phits_index_to_yt')


def upload_phits_index_to_yt(
        yt_client, yt_path_template, advq_databases, advq_phits_type,
        sandbox_list_dbs, phits_resource_class_by_chunk, make_resource_data, limit=None):
    """
    :param yt_client: Экземпляр yt.wrapper.YtClient
    :param yt_path_template: Строка, шаблон str.format
    :param advq_databases: Список строк
    :param advq_phits_type: Строка
    :param sandbox_list_dbs: projects.advq.common.list.sandbox_list_dbs или его мок
    :param phits_resource_class_by_chunk: PHITS_RESOURCE_CLASS_BY_CHUNK или его мок
    :param make_resource_data: sdk2.ResourceData или его мок
    """
    resource_cls = phits_resource_class_by_chunk[advq_phits_type]
    resource_list = [
        resource
        for by_epoch in six.itervalues(sandbox_list_dbs(resource_cls, advq_phits_type, advq_databases,
                                                        release_type=RESOURCE_RELEASE_TYPE))
        for resources in six.itervalues(by_epoch)
        for resource in resources
    ]
    if limit:
        resource_list.sort(key=operator.attrgetter('task_id'), reverse=True)
        resource_list = resource_list[:limit]
    logger.info('Totally %d released resources for %s %r',
                len(resource_list), advq_phits_type, advq_databases)
    resources_by_yt_dir = defaultdict(dict)  # {dirname: {filename: resource}}
    for resource in resource_list:
        yt_dir = yt_path_template.format(
            phits_type=resource.advq_phits_type, dbname=resource.advq_db)
        resources_by_yt_dir[yt_dir][resource.path.name] = resource

    # В словаре остались только те ресурсы, которых нет на yt или которые свежее загруженных
    # Загружаться будут в порядке времени создания
    # [(yt_dir, resource)]
    resources_to_upload = list(_filter_resources_for_uploading(yt_client, resources_by_yt_dir))
    resources_to_upload.sort(key=lambda r: r[1].created)
    logger.info('Will be uploaded %d resources', len(resources_to_upload))

    for yt_dir, resource in resources_to_upload:
        resource_yt_path = posixpath.join(yt_dir, resource.path.name)
        logger.info('Start uploading sandbox resource id=%d as %s', resource.id, resource_yt_path)
        tmp_yt_path = posixpath.join('//tmp', '{}.{}'.format(
            resource.path, ''.join(random.choice(string.ascii_letters) for _ in six.moves.range(16))))
        resource_data = make_resource_data(resource)
        with resource_data.path.open('rb') as resource_stream:
            yt_client.write_file(tmp_yt_path, resource_stream)
        with yt_client.Transaction():
            yt_client.mkdir(yt_dir, recursive=True)
            yt_client.move(tmp_yt_path, resource_yt_path)
            yt_client.set_attribute(resource_yt_path, SANDBOX_RESOURCE_ID,
                                    resource.id)
            yt_client.set_attribute(resource_yt_path, SANDBOX_CREATION_TIME,
                                    resource.created.isoformat())
        logger.info('Uploaded sandbox resource id=%d as %s', resource.id, resource_yt_path)


def _filter_resources_for_uploading(yt_client, resources_by_yt_dir):
    from yt.wrapper.errors import YtHttpResponseError
    for yt_dir, resources_by_filename in six.iteritems(resources_by_yt_dir):
        try:
            already_uploaded_files = yt_client.list(
                yt_dir, attributes=[TYPE, SANDBOX_CREATION_TIME, SANDBOX_RESOURCE_ID])
        except YtHttpResponseError as err:
            if err.error['message'].lower().strip().startswith('error resolving path'):
                yt_file_attrs = {}
            else:
                raise
        else:
            # Каждый элемент already_uploaded_files - это yt.yson.yson_types.YsonString
            yt_file_attrs = {str(f): f.attributes for f in list(already_uploaded_files)}

        for resource in six.itervalues(resources_by_filename):
            if _should_be_written(yt_dir, resource, yt_file_attrs):
                yield yt_dir, resource


def _should_be_written(yt_dir, resource, yt_file_attrs):
    yt_path = posixpath.join(yt_dir, resource.path.name)
    try:
        yt_attrs = yt_file_attrs[resource.path.name]
    except KeyError:
        logger.info('File %s not exists on YT. Will upload.', yt_path)
        return True

    if yt_attrs[TYPE] != 'file':
        raise Exception(
            "Can't write file {} because it is exists and it is a {}".format(yt_path, yt_attrs[TYPE]))

    logger.info('File %s already on YT. Will NOT upload.', yt_path)
    return False
