import logging

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.advq.artifacts import ADVQ_DELAY_MONITORING_JAR
from sandbox.projects.advq.common import validate_arcadia_rev
from sandbox.projects.advq.common.parameters import AdvqBuildBinariesResourceParameter, YtParameters, SandboxParameters, \
    releaseTo_selector
from sandbox.projects.advq.common.sandbox_utils import get_sandbox_env_from_parameters
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.advq.common.yt_utils import get_yt_env_from_parameters
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2 import ResourceData
import os


MIN_MONITORING_ARCADIA_REVISION = 3636072

CONFIG = (
    ('normal', 'http://back-normal.advq.yandex.net/advq/info?format=json'),
    # ADVQ-1897 stop spikes generation:
    # ('spikes', 'http://spikes.advq.yandex.ru/advq/info?format=json'),
    # ('video', 'http://advq-video.yandex.ru/advq/info?format=json')
)


class AdvqRunDelayMonitoring(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (SandboxJavaJdkEnvironment("11.0.2"),)

    class Parameters(sdk2.Task.Parameters):
        advq_generation_binaries = AdvqBuildBinariesResourceParameter("ADVQ generation utils resource", required=True)
        yt_parameters = YtParameters()
        sandbox_parameters = SandboxParameters()
        yav_token_secret_id = sdk2.parameters.YavSecret("YAV secret id", required=True,
                                                       description='secret should contain keys: solomon',
                                                       default='sec-01eggmyedjg4hcbtcrrd5j1dxg')
        released = releaseTo_selector(
            "Select Sandbox resources with 'released' user attribute value equal to given value",
            with_empty=True
        )

    def on_execute(self):
        validate_arcadia_rev(self.Parameters.advq_generation_binaries, [MIN_MONITORING_ARCADIA_REVISION])

        advq_binaries = ResourceData(self.Parameters.advq_generation_binaries)

        monitoring_jar = str(advq_binaries.path.joinpath(ADVQ_DELAY_MONITORING_JAR))

        env = dict(os.environ)
        env.update(get_yt_env_from_parameters(self.Parameters.yt_parameters))
        env.update(get_sandbox_env_from_parameters(self.Parameters.sandbox_parameters))
        java_cmd = ['java', '-Djava.net.preferIPv6Addresses=true',
                    '-cp', monitoring_jar, 'ru.yandex.advq.tools.monitoring.cli.Monitor',
                    '--solomon-token=' + self.Parameters.yav_token_secret_id.data()["solomon"], '--trace']

        monitoring_args = [["hitlog"], ["assoc"], ["pkz"]]
        for phits_type, backend_url in CONFIG:
            phits_type_arg = '--phits-type=' + phits_type
            released_selector_arg = '--released=' + self.Parameters.released
            backend_url_arg = '--advq-info=' + backend_url
            monitoring_args.extend([
                ['advq', phits_type_arg, backend_url_arg],
                ['yt', phits_type_arg],
                ['sandbox', phits_type_arg, released_selector_arg]
            ])

        something_failed = False

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("advq-delay-monitoring")) as pl:
            for args in monitoring_args:
                subtask = " ".join(args)
                try:
                    sp.check_call(java_cmd + args, shell=False, stdout=pl.stdout, stderr=sp.STDOUT, env=env)
                    self.set_info("SUCCESS: " + subtask)
                except sp.CalledProcessError:
                    self.set_info("FAILURE: " + subtask)
                    something_failed = True

        if something_failed:
            raise TaskError("Some delay monitoring calls failed")
