import logging

from sandbox import sdk2

from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.common.types import task as ctt

from sandbox.projects.alice_evo.common import resources
from sandbox.projects.alice_evo.common.const import EvoConstants

logger = logging.getLogger(__name__)

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class AliceEvoAnalyticsInfoBuildBinary(sdk2.Task):
    """
        Build AnalyticsInfo binary for BMV
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 minutes

        evo_in_sandbox_svn_path = sdk2.parameters.String(
            'evo_in_sandbox_svn_path',
            required=True,
            default=EvoConstants.TRUNK_SANDBOX_SVN_PATH
        )

        with sdk2.parameters.Output:
            evo_out_analytics_info_binary = sdk2.parameters.Resource(
                'evo_out_analytics_info_binary',
                resource_type=resources.AliceEvoAnalyticsInfoBinary
            )

    def on_execute(self):
        with self.memoize_stage.prepare_and_run_yamake:
            logger.info('Start prepare_and_run_yamake')
            task = sdk2.Task['KOSHER_YA_MAKE'](
                self,
                description='Build ALICE_EVO_ANALYTICS_INFO_BINARY from Arcadia source',
                arts='alice/acceptance/runners/run',
                result_rt='ALICE_EVO_ANALYTICS_INFO_BINARY',
                result_single_file=False,
                targets='alice/acceptance/runners',
                checkout_arcadia_from_url=self.Parameters.evo_in_sandbox_svn_path,
            )

            self.Context.evo_ctx_analytics_info_yamake_task_id = task.id
            task.enqueue()

            logger.info('Finish prepare_and_run_yamake')
            raise sdk2.WaitTask([task],
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.get_yamake_result:
            logger.info('Start get_yamake_result')

            yamake_task = sdk2.Task[self.Context.evo_ctx_analytics_info_yamake_task_id]

            if yamake_task.status not in Status.Group.SUCCEED:
                raise errors.TaskFailure('Failed to build AnalyticsInfo')

            self.Parameters.evo_out_analytics_info_binary = resources.AliceEvoAnalyticsInfoBinary.find(
                task=yamake_task).first().id

            logger.info('Finish get_yamake_result')
