import logging
import os
import yaml

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.alice_evo.common import resources
from sandbox.projects.alice_evo.common.const import AccessTokens

logger = logging.getLogger(__name__)


def prepare_experiments(experiments):
    if not isinstance(experiments, list):
        return []
    res = []
    for exp in experiments:
        res.append(str(exp + '=1'))
    return res


def prepare_config(config_file_path, vins_url, experiments):
    config = {
        'yt': {
            'token': 'robot-alice-priemka_yt_token',
            'mr_account': 'alice'
        },
        'yql': {
            'token': 'robot-alice-priemka_yql_token'
        },
        'sandbox': {
            'token': 'robot-alice-priemka_sandbox_token'
        },
        'soy': {
            'token': 'robot-alicesoyaccept-soy-token'
        },
        'alice_common': {
            'ya_plus_robot_token': 'robot-alice-vins_oauth_token',
            'scraper_over_yt_pool': 'alice-acceptance'
        },
        'screenshots_eval': {
            'vins_url_2': 'http://megamind.alice.yandex.net/rc/speechkit/app/pa/',
            'bass_url_2': 'http://bass.hamster.alice.yandex.net/'
        },
        'acceptance_order': [
            ['check_analytics_info']
        ],
        'check_analytics_info': {
            'input_table': '//home/alice/toloka/accept/ue2e_quasar/basket',
            'test_vins_url': str(vins_url),
            'experiments': prepare_experiments(experiments) + ['analytics_info=1', 'tunneller_analytics_info=1'],
            'revision': 6228393
        }
    }
    with open(config_file_path, 'w') as f:
        yaml.dump(config, f)


class AliceEvoAnalyticsInfoRunTests(sdk2.Task):
    """
        Run AnalyticsInfo for BMV
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60  # 10 minutes

        evo_in_vins_url = sdk2.parameters.String(
            'Vins url',
            required=True,
        )

        evo_in_experiments = sdk2.parameters.List(
            'list of experiments',
            default=[]
        )

        evo_in_analytics_info_tests_binary = sdk2.parameters.Resource(
            'AnalyticsInfo tests binary',
            resource_type=resources.AliceEvoAnalyticsInfoBinary,
            required=True
        )

        evo_in_no_exec = sdk2.parameters.Bool(
            'Do not start the graph',
            default=False
        )

        with sdk2.parameters.Group('analytics_info_cfg') as analytics_info_cfg:
            evo_in_ai_cfg_yt_token = sdk2.parameters.String(
                'evo_in_ai_cfg_yt_token',
                default='robot-alice-priemka_yt_token',
                required=True,
            )
            evo_in_ai_cfg_yt_mr_account = sdk2.parameters.String(
                'evo_in_ai_cfg_yt_mr_account',
                default='alice',
                required=True,
            )
            evo_in_ai_cfg_yql_token = sdk2.parameters.String(
                'evo_in_ai_cfg_yql_token',
                default='robot-alice-priemka_yql_token',
                required=True,
            )
            evo_in_ai_cfg_sandbox_token = sdk2.parameters.String(
                'evo_in_ai_cfg_sandbox_token',
                default='robot-alice-priemka_sandbox_token',
                required=True,
            )
            evo_in_ai_cfg_soy_token = sdk2.parameters.String(
                'evo_in_ai_cfg_soy_token',
                default='robot-alicesoyaccept-soy-token',
                required=True,
            )
            evo_in_ai_cfg_alice_common_ya_plus_robot_token = sdk2.parameters.String(
                'evo_in_ai_cfg_alice_common_ya_plus_robot_token',
                default='robot-alice-vins_oauth_token',
                required=True,
            )
            evo_in_ai_cfg_alice_common_scraper_over_yt_pool = sdk2.parameters.String(
                'evo_in_ai_cfg_alice_common_scraper_over_yt_pool',
                default='alice-acceptance',
                required=True,
            )
            evo_in_ai_cfg_screenshots_eval_vins_url_2 = sdk2.parameters.String(
                'evo_in_ai_cfg_screenshots_eval_vins_url_2',
                default='http://megamind.alice.yandex.net/rc/speechkit/app/pa/',
                required=True,
            )
            evo_in_ai_cfg_screenshots_eval_bass_url_2 = sdk2.parameters.String(
                'evo_in_ai_cfg_screenshots_eval_bass_url_2',
                default='http://bass.hamster.alice.yandex.net/',
                required=True,
            )
            evo_in_ai_cfg_check_analytics_info_input_table = sdk2.parameters.String(
                'evo_in_ai_cfg_check_analytics_info_input_table',
                default='//home/alice/toloka/accept/ue2e_quasar/basket',
                required=True,
            )
            evo_in_ai_cfg_check_analytics_info_revision = sdk2.parameters.Integer(
                'evo_in_ai_cfg_check_analytics_info_revision',
                default=6228393,
                required=True,
            )

        with sdk2.parameters.Output:
            evo_out_config_file = sdk2.parameters.Resource(
                'AnalyticsInfo tests config',
                resource_type=resources.AliceEvoAnalyticsInfoConfig)
            evo_out_tests_logs = sdk2.parameters.Resource(
                'AnalyticsInfo tests logs',
                resource_type=resources.AliceEvoAnalyticsInfoLogs)
            evo_out_analytics_info_graph = sdk2.parameters.String(
                'evo_out_analytics_info_graph',
            )

    def on_execute(self):
        self.Parameters.evo_out_config_file = resources.AliceEvoAnalyticsInfoConfig(
            self, 'AnalyticsInfo tests config file for ' + self.Parameters.description, 'config.yaml', ttl=90
        )
        config_file_path = str(sdk2.ResourceData(self.Parameters.evo_out_config_file).path)

        with self.memoize_stage.prepare_config:
            vins_url = self.Parameters.evo_in_vins_url
            experiments = self.Parameters.evo_in_experiments
            prepare_config(config_file_path, vins_url, experiments)

        with self.memoize_stage.run_analytics_info_tests:
            nirvana_oauth_token = sdk2.Vault.data(
                AccessTokens.NIRVANA_PRIEMKINS_TOKEN_OWNER,
                AccessTokens.NIRVANA_PRIEMKINS_TOKEN_NAME
            )
            os.environ['NIRVANA_TOKEN'] = nirvana_oauth_token
            cmd = [
                str(sdk2.ResourceData(self.Parameters.evo_in_analytics_info_tests_binary).path) + '/run',
                '--user-config', config_file_path,
            ]
            if self.Parameters.evo_in_no_exec:
                cmd.append('--no-exec')

            self.Parameters.evo_out_tests_logs = resources.AliceEvoAnalyticsInfoLogs(
                self, 'AnalyticsInfo tests logs' + self.Parameters.description, 'analytics_info_tests_logs.txt', ttl=90
            )
            logs_file_path = str(sdk2.ResourceData(self.Parameters.evo_out_tests_logs).path)
            logs_file = open(logs_file_path, 'w')
            sp.Popen(' '.join(cmd), shell=True, stdout=logs_file, stderr=logs_file).wait()
            logs_file.close()

            logs = open(logs_file_path, 'r').read()
            for line in logs.split():
                if 'https://nirvana.yandex-team.ru' in line:
                    self.set_info('nirvana graph url: ' + line)
                    self.Parameters.evo_out_analytics_info_graph = line

            self.set_info('full AnalyticsInfo tests stdout:\n' + logs)
        self.set_info('pid {}: finished {}'.format(os.getpid(), cmd))
