
# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.task import Status
from sandbox.projects.alice_evo.common import resources
from sandbox.projects.alice_evo.common.const import (AccessTokens,
                                                     AnalyticsInfoConstants,
                                                     EvoConstants)
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox

import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as sdk_environments

logger = logging.getLogger(__name__)

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class AliceEvoAnalyticsInfoWrapper(sdk2.Task):
    """
        Handle AnalyticsInfo test launch for BMV
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [sdk_environments.PipEnvironment('startrek_client',
                                                        version='2.3.0',
                                                        custom_parameters=['--upgrade-strategy only-if-needed']), ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min

        evo_in_sandbox_svn_path = sdk2.parameters.String(
            'evo_in_sandbox_svn_path',
            required=True,
            default=EvoConstants.TRUNK_SANDBOX_SVN_PATH
        )
        evo_in_megamind_url = sdk2.parameters.String(
            'evo_in_megamind_url',
            required=True,
            default=EvoConstants.MEGAMIND_HAMSTER_URL
        )
        evo_in_tag_num = sdk2.parameters.String(
            'evo_in_tag_num',
            required=True,
            default="0"
        )
        evo_in_build_id = sdk2.parameters.String(
            'evo_in_build_id',
            required=True,
            default="vins-xx-xx"
        )
        evo_in_force_start = sdk2.parameters.Bool(
            'evo_in_force_start',
            required=True,
            default=False
        )
        evo_in_st_ticket = sdk2.parameters.String(
            'evo_in_st_ticket',
            required=True,
            default=EvoConstants.PLACEHOLDER_ST_TICKET,
        )

        with sdk2.parameters.Output:
            evo_out_analytics_info_binary = sdk2.parameters.Resource(
                'evo_out_analytics_info_binary',
                resource_type=resources.AliceEvoAnalyticsInfoBinary
            )
            evo_out_analytics_info_graph = sdk2.parameters.String(
                "evo_out_analytics_info_graph"
            )

    def on_execute(self):
        logger.info("[INFO] Started on_execute")

        with self.memoize_stage.prepare_and_run_build:
            logger.info('Start prepare_and_run_build')

            self.Context.evo_ctx_need_start = False
            if self.Parameters.evo_in_force_start or (self.Parameters.evo_in_tag_num == "1"):
                self.Context.evo_ctx_need_start = True

            task = sdk2.Task['ALICE_EVO_ANALYTICS_INFO_BUILD_BINARY'](
                self,
                description='Created by wrapper task',
                evo_in_sandbox_svn_path=self.Parameters.evo_in_sandbox_svn_path,
            )

            self.Context.evo_ctx_analytics_info_build_task_id = task.id
            task.enqueue()

            logger.info('Finish prepare_and_run_build')
            raise sdk2.WaitTask([task],
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.get_build_result_and_run:
            logger.info('Start get_build_result')

            build_task = sdk2.Task[self.Context.evo_ctx_analytics_info_build_task_id]

            if build_task.status not in Status.Group.SUCCEED:
                raise errors.TaskFailure('Failed to build AnalyticsInfo')

            self.Parameters.evo_out_analytics_info_binary = build_task.Parameters.evo_out_analytics_info_binary

            logger.info('Finish get_build_result')

            logger.info('Start run')
            task = sdk2.Task['ALICE_EVO_ANALYTICS_INFO_RUN_TESTS'](
                self,
                description='Created by wrapper task',
                evo_in_vins_url=self.Parameters.evo_in_megamind_url,
                evo_in_analytics_info_tests_binary=self.Parameters.evo_out_analytics_info_binary,
                evo_in_no_exec=not self.Context.evo_ctx_need_start,
            )

            self.Context.evo_ctx_analytics_info_run_task_id = task.id
            task.enqueue()

            logger.info('Finish run')
            raise sdk2.WaitTask([task],
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.get_run_results:
            logger.info('Start get_run_result')

            run_task = sdk2.Task[self.Context.evo_ctx_analytics_info_run_task_id]

            if run_task.status not in Status.Group.SUCCEED:
                raise errors.TaskFailure('Failed to run AnalyticsInfo')

            self.Parameters.evo_out_analytics_info_graph = run_task.Parameters.evo_out_analytics_info_graph

            comment = AliceEvoAnalyticsInfoWrapperHelper.generate_comment(
                self.Parameters.evo_in_build_id,
                self.Context.evo_ctx_need_start,
                self.Parameters.evo_out_analytics_info_graph,
            )

            from startrek_client import Startrek
            st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                                 useragent=AccessTokens.ST_USERAGENT)

            StartrekToolbox.append_startrek_comment(
                st_client,
                comment,
                self.Parameters.evo_in_st_ticket,
                AnalyticsInfoConstants.ST_COMMENT_TITLE,
                False
            )

            logger.info('Finish get_run_result')


class AliceEvoAnalyticsInfoWrapperHelper(object):
    @staticmethod
    def generate_comment(build_id,
                         need_start,
                         nirvana_url_combo):
        """
        Generate comment
        with cut, summary and link to report

        """
        comment = "[" + datetime.today().strftime("%Y-%m-%d %H:%M:%S") + "] " + \
                  "AnalyticsInfo for " + build_id + " "

        if need_start:
            comment += "has been created and started"
        else:
            comment += "has been prepared (however not automatically started)"

        comment += ": " + nirvana_url_combo + "\n"

        return comment
