# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.projects.alice_evo.common.const import AccessTokens, EvoConstants
from sandbox.projects.alice_evo.common.release_machine import ReleaseMachineToolbox
from sandbox.projects.alice_evo.common.yappy import YappyToolbox

import sandbox.projects.release_machine.core.task_env as task_env

logger = logging.getLogger(__name__)


class AliceEvoAppHostContext(sdk2.Task):
    '''
        Context for Alice Evo tests in release machine
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = \
            task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min
        evo_in_branch_num = sdk2.parameters.String(
            'evo_in_branch_num',
            required=True
        )
        evo_in_tag_num = sdk2.parameters.String(
            'evo_in_tag_num',
            required=True
        )
        with sdk2.parameters.Output:
            evo_out_branch_num = sdk2.parameters.String(
                'evo_out_branch_num'
            )
            evo_out_tag_num = sdk2.parameters.String(
                'evo_out_tag_num'
            )
            evo_out_component_name = sdk2.parameters.String(
                'evo_out_component_name'
            )
            evo_out_beta_name = sdk2.parameters.String(
                'evo_out_beta_name'
            )
            evo_out_http_adapter_url = sdk2.parameters.String(
                'evo_out_http_adapter_url'
            )
            evo_out_yappy_page = sdk2.parameters.String(
                'evo_out_yappy_page'
            )
            evo_out_release_ticket = sdk2.parameters.String(
                'evo_out_release_ticket'
            )
            evo_out_beta_slots = sdk2.parameters.List(
                'evo_out_beta_slots'
            )

    def on_execute(self):
        logger.info("[INFO] Start task")
        # branch_num
        self.Parameters.evo_out_branch_num = self.Parameters.evo_in_branch_num
        self.Context.evo_ctx_branch_num = self.Parameters.evo_in_branch_num

        # tag_num
        self.Parameters.evo_out_tag_num = self.Parameters.evo_in_tag_num
        self.Context.evo_ctx_tag_num = self.Parameters.evo_in_tag_num

        # component_name
        self.Context.evo_ctx_component_name = EvoConstants.ALICE_GRAPHS_RM_NAME
        self.Parameters.evo_out_component_name = EvoConstants.ALICE_GRAPHS_RM_NAME

        # beta_name
        beta_prefix = "app-host-alice"
        beta_name = \
            YappyToolbox.get_beta_name(
                beta_prefix,
                self.Parameters.evo_in_branch_num,
                self.Parameters.evo_in_tag_num
            )
        self.Context.evo_ctx_beta_name = beta_name
        self.Parameters.evo_out_beta_name = beta_name

        # beta_slots
        # Yappy URL, e.g.: http://app-host-alice-3-1.yappy.beta.yandex.ru/speechkit/app/pa/
        yappy_token = sdk2.Vault.data(AccessTokens.YAPPY_TOKEN_OWNER, AccessTokens.YAPPY_TOKEN_NAME)
        beta_slots = YappyToolbox.get_yappy_slots(
            self.Context.evo_ctx_beta_name,
            yappy_token,
            beta_prefix
        )
        self.Context.evo_ctx_beta_slots = beta_slots
        self.Parameters.evo_out_beta_slots = beta_slots

        # http_adapter_url
        beta_url = \
            YappyToolbox.get_megamind_speechkit_app_pa_url(
                beta_prefix,
                self.Parameters.evo_in_branch_num,
                self.Parameters.evo_in_tag_num
            )
        self.Context.evo_ctx_http_adapter_url = beta_url
        self.Parameters.evo_out_http_adapter_url = beta_url

        # yappy_page
        yappy_page = \
            YappyToolbox.get_yappy_page(
                beta_prefix,
                self.Parameters.evo_in_branch_num,
                self.Parameters.evo_in_tag_num
            )
        self.Parameters.evo_out_yappy_page = yappy_page
        self.Context.evo_ctx_yappy_page = yappy_page

        # release_ticket
        release_ticket = \
            ReleaseMachineToolbox.get_release_ticket_for_branch(
                self.Context.evo_ctx_component_name,
                self.Context.evo_ctx_branch_num
            )
        self.Parameters.evo_out_release_ticket = release_ticket
        self.Context.evo_ctx_release_ticket = release_ticket

        logger.info("[INFO] Finish task")
