# -*- coding: utf-8 -*-

import logging
import requests

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.vins.common.resources import VinsPackage
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox.projects.alice_evo.common.const import EvoConstants
from sandbox.projects.alice_evo.common.nanny import NannyToolbox
from sandbox.projects.alice_evo.common.release_machine import ReleaseMachineToolbox
from sandbox.projects.alice_evo.common.svn import SvnToolbox

logger = logging.getLogger(__name__)

MEGAMIND_TIMEOUT = 5.0


class AliceEvoContextFast(sdk2.Task):
    """
        Fast context for Alice Evo tests in release machine

    """

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = \
            task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min
        evo_in_branch_num = sdk2.parameters.String(
            "evo_in_branch_num",
            required=True
        )
        evo_in_tag_num = sdk2.parameters.String(
            "evo_in_tag_num",
            required=True
        )
        evo_in_component_name = sdk2.parameters.String(
            "evo_in_component_name",
            default=EvoConstants.VINS_RM_NAME
        )
        with sdk2.parameters.Output:
            evo_out_release_id = sdk2.parameters.String(
                "evo_out_release_id"
            )
            evo_out_release_ticket = sdk2.parameters.String(
                "evo_out_release_ticket"
            )
            evo_out_current_prod_vins_package = sdk2.parameters.Resource(
                "evo_out_current_prod_vins_package",
                resource_type=VinsPackage
            )
            evo_out_sandbox_svn_path = sdk2.parameters.String(
                "evo_out_sandbox_svn_path"
            )
            evo_out_sandbox_stable_svn_path = sdk2.parameters.String(
                "evo_out_sandbox_stable_svn_path"
            )
            evo_out_shell_svn_path = sdk2.parameters.String(
                "evo_out_shell_svn_path"
            )

    def _get_stable_tag(self):
        resp = requests.get(
            'http://megamind.alice.yandex.net/version_json',
            headers={
                'Content-Type': 'application/json',
            },
            timeout=MEGAMIND_TIMEOUT
        )
        resp.raise_for_status()

        logger.info('Version json from Megamind received')
        stable_tag = resp.json().get('tag')

        if stable_tag is None:
            raise errors.TaskError("Can't get tag from /version_json response: %s" % resp.json())

        return stable_tag

    def on_execute(self):
        logger.info("[INFO] Start task")

        self.Context.evo_ctx_branch_num = self.Parameters.evo_in_branch_num
        self.Context.evo_ctx_tag_num = self.Parameters.evo_in_tag_num
        self.Context.evo_ctx_component_name = self.Parameters.evo_in_component_name

        self.Parameters.evo_out_release_id = "{}-{}-{}".format(
            self.Context.evo_ctx_component_name,
            self.Context.evo_ctx_branch_num,
            self.Context.evo_ctx_tag_num
        )
        self.Context.evo_ctx_release_id = \
            self.Parameters.evo_out_release_id

        self.Parameters.evo_out_release_ticket = \
            ReleaseMachineToolbox.get_release_ticket_for_branch(
                self.Context.evo_ctx_component_name,
                self.Context.evo_ctx_branch_num
            )
        self.Context.evo_ctx_release_ticket = \
            self.Parameters.evo_out_release_ticket

        # e.g.: arcadia:/arc/tags/vins/stable-41-14/arcadia
        self.Parameters.evo_out_sandbox_svn_path = \
            SvnToolbox.get_tag_sandbox_svn_path(
                self.Context.evo_ctx_branch_num,
                self.Context.evo_ctx_tag_num,
                EvoConstants.VINS_RM_NAME
            )

        stable_tag = self._get_stable_tag()
        # e.g.: arcadia:/arc/tags/vins/stable-40-5/arcadia
        self.Parameters.evo_out_sandbox_stable_svn_path = \
            'arcadia:/arc/tags/%s/arcadia' % stable_tag

        # e.g.: svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-41-14/arcadia
        self.Parameters.evo_out_shell_svn_path = \
            SvnToolbox.get_tag_svn_ssh_url(
                self.Context.evo_ctx_branch_num,
                self.Context.evo_ctx_tag_num,
                EvoConstants.VINS_RM_NAME
            )

        self.Parameters.evo_out_current_prod_vins_package = \
            NannyToolbox.get_prod_vins_package()
        logger.info("[INFO] Finish task")
