# -*- coding: utf-8 -*-

import logging
import os

from os.path import join as pj
from sandbox.sdk2.vcs.svn import Svn
from sandbox import sdk2
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox
from sandbox.projects.alice_evo.common.svn import SvnToolbox


logger = logging.getLogger(__name__)


class AliceEvoDiffConfig(sdk2.Task):
    """
        Make config diff for Alice Megamind release
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min
        branch_num = sdk2.parameters.String(
            "branch_num",
            required=False
        )
        tag_num = sdk2.parameters.String(
            "tag_num",
            required=False
        )
        component_name = sdk2.parameters.String(
            "component_name",
            required=False
        )
        release_ticket = sdk2.parameters.String(
            "release_ticket",
            required=True,
            default="ALICERELTEST-70"
        )
        notify_first_tag = sdk2.parameters.Bool(
            "notify_first_tag",
            required=True,
            default=True
        )
        notify_diff_between_tags = sdk2.parameters.Bool(
            "notify_diff_between_tags",
            required=True,
            default=False
        )
        with sdk2.parameters.Output:
            diff_id = sdk2.parameters.String(
                "diff_id"
            )

    def on_execute_old_2(self):
        # url_1 = "svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-41-14/arcadia/alice"
        # url_1 = "svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-41-15/arcadia/alice"
        url_1 = "svn+ssh://arcadia.yandex.ru/arc/branches/vins/vins-40/arcadia/alice"
        url_2 = "svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-41-15/arcadia/alice"
        diff_text = self.get_diff(url_1, url_2)
        logger.info(diff_text)
        logger.info(str(len(diff_text)))

    def on_execute(self):
        # current
        cur_id = self.generate_id(self.Parameters.branch_num,
                                  self.Parameters.tag_num)
        cur_url = \
            SvnHelper.get_tag_configs_url(self.Parameters.branch_num,
                                          self.Parameters.tag_num,
                                          self.Parameters.component_name)

        # prev
        prev_id = None
        prev_url = None
        if self.Parameters.tag_num == "1":
            branch_num_int = int(self.Parameters.branch_num)
            prev_id = self.generate_id(str(branch_num_int - 1))
            prev_url = \
                SvnHelper.get_branch_configs_url(str(branch_num_int - 1),
                                                 self.Parameters.component_name)
        else:
            tag_num_int = int(self.Parameters.tag_num)
            prev_id = self.generate_id(self.Parameters.branch_num,
                                       str(tag_num_int - 1))
            prev_url = \
                SvnHelper.get_tag_configs_url(self.Parameters.branch_num,
                                              str(tag_num_int - 1),
                                              self.Parameters.component_name)

        diff_text = self.get_diff(prev_url, cur_url)
        logger.info(diff_text)
        logger.info(str(len(diff_text)))
        if len(diff_text) > 0:
            logger.info("Got some diff")
        else:
            logger.info("No diff")

        diff_id = prev_id + "_vs_" + cur_id + ".diff"
        self.Parameters.diff_id = diff_id

        if self.Parameters.notify_first_tag and (self.Parameters.tag_num == "1"):
            logger.info("need to post result to gist if possible")
            if len(diff_text) == 0:
                logger.info("nothing to post")
                comment_text = \
                    "There is no difference in configs diff " + diff_id
                StartrekToolbox.post_single_comment(
                    comment_text,
                    self.Parameters.release_ticket
                )
            else:
                logger.info("diff_text is available")
                comment_text = (
                    "Configs diff **{0}** is ready:\n<{{diff\n%%(diff)\n{1}\n%%\n}}>\n"
                ).format(diff_id, diff_text)

                StartrekToolbox.post_single_comment(
                    comment_text,
                    self.Parameters.release_ticket
                )

        if len(diff_text) > 0:
            logger.info("nice")
            # logger.info(str(len(diff_text)))
            # self.paste_diff(diff_text)

    def get_diff(self, arcadia_url_left, arcadia_url_right):
        diff_text = ""
        for part_name in ["production"]:
            diff_text += self.get_diff_part(arcadia_url_left,
                                            arcadia_url_right,
                                            part_name)
        return diff_text

    def get_diff_part(self, arcadia_url_left, arcadia_url_right, part_name):
        arcadia_url_left += part_name
        arcadia_url_right += part_name
        return Svn.diff([arcadia_url_left, arcadia_url_right])

    def download_config(self, arcadia_url, dir_name):
        dst_path = pj(os.getcwd(), dir_name)
        # os.chdir(dst_path)
        # sdk2.svn
        # https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/sdk2/vcs/svn.py?rev=6582361#L542
        sdk2.svn.Arcadia.checkout(arcadia_url,
                                  dst_path)

    def generate_id(self, branch_num, tag_num=None):
        release_id = self.Parameters.component_name + "-" + branch_num
        if tag_num is not None:
            release_id = release_id + "-" + tag_num
        return release_id


class SvnHelper(object):
    @staticmethod
    def get_branch_configs_url(branch_num, component_name):
        """
        Get svn+ssh arcadia url to alice/megamind/configs path for branch
        e.g.: "svn+ssh://arcadia.yandex.ru/arc/branches/vins/vins-40/arcadia/alice/megamind/configs"
        e.g.: "svn+ssh://arcadia.yandex.ru/arc/branches/apphost/conf/alice/stable-40/arcadia/alice/megamind/configs"

        """

        svn_ssh_url = SvnToolbox.get_branch_svn_ssh_url(branch_num, component_name)
        return svn_ssh_url + "/alice/megamind/configs/"

    @staticmethod
    def get_tag_configs_url(branch_num, tag_num, component_name):
        """
        Get svn+ssh arcadia url to alice/megamind/configs path for tag
        e.g.: "svn+ssh://arcadia.yandex.ru/arc/tags/vins/stable-41-15/arcadia/alice/megamind/configs"
        e.g.: "svn+ssh://arcadia.yandex.ru/arc/tags/apphost/conf/alice/stable-41-15/arcadia/alice/megamind/configs"

        """

        svn_ssh_url = SvnToolbox.get_tag_svn_ssh_url(branch_num, tag_num, component_name)
        return svn_ssh_url + "/alice/megamind/configs/"
