# -*- coding: utf-8 -*-
import json
import re
import os.path
import time
import urllib

import requests

import sandbox.common.types.task as ctt
import sandbox.projects.common.build.parameters as build_params
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import sdk2
from sandbox import common
from sandbox.projects.alice_evo.AliceEvoIntegrationTests import AliceEvoIntegrationTests
from sandbox.projects.alice_evo.AliceEvoMarkerTests import AliceEvoMarkerTests
from sandbox.projects.alice_evo.CallEvoTestOwners import CallEvoTestOwners


FAIL_COLOR = '#b7141e'
GOOD_COLOR = '#457b23'

FAIL_STATUS = 'FAIL'
GOOD_STATUS = 'GOOD'

LINK_TEMPLATE = '''
    <div>
    <span style="color:#424242;font-weight:bold;">{name}:</span>
    <span style="color:#457b23;"> </span>
    <a href="{href}" target="_blank" style="color:#134eb3;">{href_text}</a>
    </div>
'''
ST_COMMENT_TITLE_TEMPLATE = '===={title}===='


SUBTASK_CLS = {
    'ALICE_EVO_INTEGRATION_TESTS': AliceEvoIntegrationTests,
    'ALICE_EVO_MARKER_TESTS': AliceEvoMarkerTests,
}


def _create_link(name, href):
    href_text = urllib.unquote(href).decode('utf8') if href else 'Ссылка отсутствует'
    return LINK_TEMPLATE.format(name=name, href=href, href_text=href_text)


def _make_test_name(subtask_name):
    def _capitalize(word):
        return word.capitalize() if word != 'EVO' else word
    return ' '.join([_capitalize(w) for w in subtask_name.split('_')])


def _iter_snippet(snippets):
    begin = 0
    end = 0
    index = 0
    while begin >= 0:
        while snippets.find('[[bad]]', begin, end) == -1:
            end = snippets.find('[[unimp]]', end + 1)
        yield index, snippets[begin:end if end > 0 else len(snippets)]
        begin = end
        index += 1


def _create_colorfull_snippet(snippet):
    html = []
    tokens = [_.split(']]') for _ in snippet.split('[[')[1:]]
    try:
        for name, value in tokens:
            if name == 'unimp':
                html.append('<div><span style="color:#999999;font-weight:bold;">{}</span>'.format(value))
            elif name == 'alt1':
                html.append('<span style="color:#0e717c;">{}</span></div>'.format(value))
            elif name == 'alt2':
                html.append('<span style="color:#550088;">{}</span>'.format(value))
            elif name == 'bad':
                error = '<div>{}</div>'.format('</div><div>E   '.join(value.split('E ')))
                html.append('<span style="color:#b7141e;">{}</span>'.format(error))
            else:
                html.append(value)
    except:
        html.append(snippet)
    return ''.join(html)


def _create_colorfull_metric_snippet(snippet):
    html = []
    tokens = [_.split(']]') for _ in snippet.split('[[')[1:]]
    colors = {'unimp': '#999999;font-weight:bold', 'bad': FAIL_COLOR, 'good': GOOD_COLOR, 'warn': '#fd7b08'}
    try:
        for name, value in tokens:
            if name in colors:
                html.append('<span style="color:{};">{}</span>'.format(colors[name], value))
            else:
                html.append(value)
    except:
        html.append(snippet)
    return ''.join(html)


def _get_html_resource_path(task):
    html_resources = sdk2.Resource['BUILD_OUTPUT_HTML'].find(task=task).limit(10)
    for r in html_resources:
        path = str(sdk2.ResourceData(r).path)
        if re.match('output_\d+_1.html', os.path.basename(path)):
            return r.http_proxy


def _make_setrace_url(log_url):
    def _find_uuid(text):
        expr = re.findall(r'Uuid of fake-user: \b[a-z0-9]+\b', text)
        return expr[0].strip().split()[-1] if expr else None

    try:
        log_uuid = _find_uuid(requests.get(log_url, headers={'Accept': 'text/html'}).text)
    except:
        return 'bad url given'

    if not log_uuid:
        return 'setrace url not found'
    return 'https://setrace.yandex-team.ru/ui/alice/sessionsList?trace_by={uuid}'.format(uuid=log_uuid)


def _is_release_error(text):
    return '__tests__.conftest.ReleaseBugError' in text


def _make_html_report(lines):
    if not lines:
        return 'Can\'t find test suite in test results.'
    return '<span style="font-family:monospace;font-size:12px;">{}</span>'.format(''.join(lines))


class AliceEvoIntegrationTestsWrapper(sdk2.Task):
    """
    Wrapper to build and run Alice EVO Tests in release machine
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment('startrek_client', version='2.3.0', custom_parameters=['--upgrade-strategy only-if-needed']),
        ]

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.String('Subtask type', required=True, multiline=True) as subtask_type:
            subtask_type.values.ALICE_EVO_INTEGRATION_TESTS = subtask_type.Value('ALICE_EVO_INTEGRATION_TESTS', default=True)
            subtask_type.values.ALICE_EVO_MARKER_TESTS = 'ALICE_EVO_MARKER_TESTS'
        checkout_arcadia_from_url = build_params.ArcadiaUrl()
        arcadia_patch = build_params.ArcadiaPatch()
        test_filters = build_params.TestFilters()

        with sdk2.parameters.Group('Run test parameters') as param_block:
            uniproxy_url = sdk2.parameters.String('Uniproxy url')
            megamind_url = sdk2.parameters.String('Megamind (aka vins) url')
            experiments = sdk2.parameters.String('Experiments (e.g., "disregard_uaas: None, mm_enable_protocol_scenario=Vins, market_disable")')
            i_want_to_kill_prod = sdk2.parameters.Bool('I want to kill production (allow production urls)')
            enable_stats = sdk2.parameters.Bool('Collect and send EVO stats to ydb', default=True)
            flaky_runs = sdk2.parameters.Integer('Max flaky test runs')
            repeat_failed_test = sdk2.parameters.Bool('Repeat failed tests on Hamster', default=True)
            dry_run = sdk2.parameters.Bool('Dry run', default=False)
            ab_testid = sdk2.parameters.String('AB testid')
            fail_threshold = sdk2.parameters.Integer('Min failed tests to fail task', default=0)

        with sdk2.parameters.Group('YAV_TOKEN from Sandbox Vault') as yav_block:
            yav_token_name = sdk2.parameters.String('Name', required=True, default='yav-alice-integration-tests-token')
            yav_token_owner = sdk2.parameters.String('Owner', required=True, default='mihajlova')

        with sdk2.parameters.Group('Release info') as release_block:
            release_ticket = sdk2.parameters.String('ALICERELEASE ticket')
            beta_name = sdk2.parameters.String('Yappy beta')
            launch_type = sdk2.parameters.String('Launch type (e.g., vins, hollywood, hamster)')
            branch_number = sdk2.parameters.Integer('Branch number', default=None)
            tag_number = sdk2.parameters.Integer('Tag number', default=None)
            run_call_owner_subtask = sdk2.parameters.Bool('Run CallEvoTestOwners subtask (required release_ticket)', default=False)
            with run_call_owner_subtask.value[True]:
                release_fail_threshold = sdk2.parameters.Integer('Min failed release tests for achtung message', default=100)

        with sdk2.parameters.Group('Solomon settings') as solomon_block:
            enable_solomon = sdk2.parameters.Bool('Collect and send test metrics to Solomon', default=False)
            with enable_solomon.value[True]:
                solomon_cluster = sdk2.parameters.String('Solomon cluster')
                solomon_service = sdk2.parameters.String('Solomon service')

    class Context(sdk2.Task.Context):
        subtask_id = None
        head_report = None
        html_result_path = None
        error_report = None
        release_error_report = None

    def on_execute(self):
        if self.Parameters.dry_run:
            return

        with self.memoize_stage.run_tests(commit_on_entrance=False):
            subtask = SUBTASK_CLS[self.Parameters.subtask_type](
                self,
                owner=self.Parameters.owner,
                checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
                arcadia_patch=self.Parameters.arcadia_patch,
                test_filters=self.Parameters.test_filters,
                disable_test_timeout=True,
                env_vars='YAV_TOKEN=\'$(vault:value:{}:{})\''.format(self.Parameters.yav_token_owner, self.Parameters.yav_token_name),
                test_params=self._make_test_params(),
            ).enqueue()
            self.Context.subtask_id = subtask.id
            raise sdk2.WaitTask(subtask, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

        subtask = self.find(id=self.Context.subtask_id).first()

        with self.memoize_stage.fill_context(commit_on_entrance=False):
            self.Context.html_result_path = _get_html_resource_path(task=subtask)

            testenv_result = self._load_testenv_result(sdk2.Resource['TEST_ENVIRONMENT_JSON_V2'].find(task=subtask).first())
            error_tests = self._get_error_tests(testenv_result)
            error_report = self._create_error_report(error_tests)
            release_errors = list(filter(_is_release_error, error_report))

            self.Context.head_report = self._create_head_report(testenv_result)
            self.Context.error_report = _make_html_report(error_report)
            self.Context.release_error_report = _make_html_report(release_errors)

            self.Context.status = ctt.Status.SUCCESS
            self.Context.message = 'Subtask {} ({}) was finished with the status of {}.\n'.format(subtask.type, subtask.id, subtask.status)
            if self.Parameters.fail_threshold:
                fail_count = len(error_report or [])
                if fail_count >= self.Parameters.fail_threshold:
                    self.Context.status = subtask.status
                    self.Context.message += 'Failed by error threshold {} ({}).\n'.format(self.Parameters.fail_threshold, fail_count)
            elif self.Parameters.repeat_failed_test:
                if release_errors:
                    self.Context.status = subtask.status
                    self.Context.message += 'Failed by release errors ({}).\n'.format(len(release_errors))
            else:
                self.Context.status = subtask.status
                self.Context.message += 'Failed by subtask.\n'

            self._send_startrek_notification(self._create_startrek_message(self.Context.status))
            raise sdk2.WaitTime(1)

        with self.memoize_stage.call_owners(commit_on_entrance=False):
            if self.Parameters.run_call_owner_subtask:
                subtask_call_test_owners = CallEvoTestOwners(
                    self,
                    owner=self.Parameters.owner,
                    task_id=self.id,
                    release_ticket=self.Parameters.release_ticket,
                    test_name=_make_test_name(self.Parameters.subtask_type),
                    only_release=self.Parameters.repeat_failed_test,
                    fail_threshold=self.Parameters.release_fail_threshold,
                    release_version='{}-{}'.format(self.Parameters.branch_number, self.Parameters.tag_number),
                ).enqueue()
                raise sdk2.WaitTask(subtask_call_test_owners, list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK), True)

        if self.Context.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure(self.Context.message)

    @sdk2.report(title='Error test results')
    def error_report(self):
        return self.Context.error_report or 'No report discovered in context'

    @sdk2.report(title='Release errors')
    def release_error_report(self):
        return self.Context.release_error_report or 'No release errors discovered in context'

    @sdk2.header()
    def head(self):
        task_link = common.utils.get_task_link(self.id)
        return '''
            <span style="font-family:monospace;font-size:12px;">
            {report}{release_error_report}{error_report}{full_report}
            </span>
        '''.format(
            report=self.Context.head_report,
            release_error_report=_create_link(name='Release error test report', href=task_link + '/release_error_report'),
            error_report=_create_link(name='Error test report', href=task_link + '/error_report'),
            full_report=_create_link(name='Full test report', href=self.Context.html_result_path),
        )

    def _make_test_params(self):
        def _make_param(k, v):
            return '{}=\'{}\''.format(k, v) if v else None

        test_params = [
            _make_param('uniproxy-url', self.Parameters.uniproxy_url),
            _make_param('vins-url', self.Parameters.megamind_url),
            _make_param('yes-i-want-to-kill-the-production', self.Parameters.i_want_to_kill_prod),
            _make_param('exps', self.Parameters.experiments),
            _make_param('repeat-failed-test-on-hamster', self.Parameters.repeat_failed_test),
            _make_param('stats-timestamp', int(time.time())),
            _make_param('flaky-runs', self.Parameters.flaky_runs),
        ]
        if self.Parameters.enable_stats:
            test_params.extend([
                _make_param('enable-stats', True),
                _make_param('sandbox-username', self.author),
                _make_param('sandbox-task', common.utils.get_task_link(self.id)),
                _make_param('sandbox-launch-type', self.Parameters.launch_type),
                _make_param('sandbox-branch-number', self.Parameters.branch_number),
                _make_param('sandbox-tag-number', self.Parameters.tag_number),
            ])
        if self.Parameters.enable_solomon:
            test_params.extend([
                _make_param('solomon-cluster', self.Parameters.solomon_cluster),
                _make_param('solomon-service', self.Parameters.solomon_service),
            ])
        return ' '.join([_ for _ in test_params if _])

    def _create_startrek_message(self, status):
        title = 'Tests with {beta_name} **!!({color}){status}!!**'.format(
            beta_name=self.Parameters.beta_name or self.Parameters.megamind_url,
            color='green' if status in ctt.Status.Group.SUCCEED else 'red',
            status=status,
        )
        if self.Parameters.ab_testid:
            title += ' for AB testid {ab_testid}'.format(ab_testid=self.Parameters.ab_testid)
        task_link = common.utils.get_task_link(self.id)
        content = '%%{uniproxy_url}%%\n<#{report}#>\n{release_error_report}\n{error_report}\n{full_report}'.format(
            uniproxy_url=self.Parameters.uniproxy_url,
            report=self.Context.head_report,
            release_error_report='Release error test report: {href}/release_error_report'.format(href=task_link),
            error_report='Error test report: {href}/error_report'.format(href=task_link),
            full_report='Full test report: {href}'.format(href=self.Context.html_result_path),
        )
        return '<{{{title}\n{content}\n}}>\n\n'.format(title=title, content=content)

    def _send_startrek_notification(self, text):
        if not self.Parameters.release_ticket:
            return

        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('robot_bassist_startrek_oauth_token'), useragent='robot-bassist')
        issue = st_client.issues[self.Parameters.release_ticket]

        comment = None
        title = ST_COMMENT_TITLE_TEMPLATE.format(title=_make_test_name(self.Parameters.subtask_type))
        for issue_comment in issue.comments.get_all():
            if issue_comment.text.startswith(title):
                comment = issue_comment
                break

        comment_text = (comment.text + text if comment else '{}\n\n{}'.format(title, text))

        if comment:
            issue.comments[comment.id].update(text=comment_text)
        else:
            issue.comments.create(text=comment_text)

    @staticmethod
    def _load_testenv_result(resource):
        path = str(sdk2.ResourceData(resource).path)
        with open(path) as stream:
            return json.load(stream)['results']

    def _create_head_report(self, testenv_result):
        html = []
        suite_results = [_ for _ in testenv_result or {} if _.get('suite', False)]
        for s in suite_results:
            html.append(self._create_suite_metrics_html(
                s['path'], s['name'], s['status'], s['metrics'][' test_count'], s['rich-snippet']
            ))
        return ''.join(html)

    @staticmethod
    def _get_error_tests(testenv_result):
        suite_results = [_ for _ in testenv_result or {} if _.get('suite', False)]
        suite_id = [_['id'] for _ in suite_results if _['name'] == 'py3test']
        if not suite_id:
            return

        suite_id = suite_id[0]
        return [_ for _ in testenv_result if _.get('suite_id') == suite_id and _.get('status') == 'FAILED']

    def _create_error_report(self, error_tests):
        if error_tests is None:
            return []

        html = []
        if not error_tests:
            html.append('<span style="color:{color};font-weight:bold;">All test are OK</span>'.format(color=GOOD_COLOR))

        for test in error_tests:
            error_snippet = self._create_error_snippet(test['rich-snippet'], test['links']['log'][0])
            html.append('''
                <div>
                [<span style="color:{color};">{status}</span>]
                <span style="color:#424242;font-weight:bold;">{name}</span>::{subtest_name}
                [tags: <span style="color:#424242;font-weight:bold;">{tags}</span>]
                {error_snippet}
                {logsdir}
                </div>
            '''.format(
                color=FAIL_COLOR,
                status=FAIL_STATUS.lower(),
                name=test['name'],
                subtest_name=test['subtest_name'],
                tags=', '.join([_ for _ in test['tags'] if not _.startswith('ya')]),
                error_snippet=''.join(error_snippet),
                logsdir=_create_link(name='Logsdir', href=test['links']['logsdir'][0]),
            ))
        return html

    @staticmethod
    def _create_error_snippet(snippets, log_link):
        html = []
        log_link = log_link[:-2]
        for index, snippet in _iter_snippet(snippets):
            log_href = log_link + ('.{}'.format(index) if index else '')
            html.append('<div>{snippet}{setrace}{log}</div>'.format(
                snippet=_create_colorfull_snippet(snippet),
                setrace=_create_link(name='Setrace', href=_make_setrace_url(log_href)),
                log=_create_link(name='Log', href=log_href),
            ))
        return html

    @staticmethod
    def _create_suite_metrics_html(path, name, status, total_count, snippet):
        return '''
            <div>
            <span style="color:#424242;font-weight:bold;">{path}</span>
            <<span style="color:#999999;font-weight:bold;">{type}</span>>
            <span style="display:block;">
            ------- <span style="color:{color};">{status}</span>: {snippet}
            </span>
            <span style="display:block;">Total {total_count} tests.</span>
            </div>
        '''.format(
            path=path,
            type=name,
            color=GOOD_COLOR if status == 'OK' else FAIL_COLOR,
            status=GOOD_STATUS if status == 'OK' else FAIL_STATUS,
            snippet=_create_colorfull_metric_snippet(snippet),
            total_count=total_count
        )
