import logging
import os
import subprocess
import shutil

from os.path import join as pj

from sandbox import sdk2
from sandbox.common import errors as common_errors
from sandbox.common.types import task as ctt
from sandbox.projects.vins.common.resources import VinsToolPaResource
from sandbox.projects.alice_evo.common.const import (AccessTokens,
                                                     EvoConstants,
                                                     LegacyConstants)
from sandbox.projects.alice_evo.common.gist import GistToolbox
from sandbox.projects.alice_evo.common.misc import MiscToolbox
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox
from sandbox.projects.vins.common.resources import VinsPackage
import sandbox.sandboxsdk.environments as sdk_environments

JSON_OUTPUT_FILE_NAME = \
    'legacy_tests_results_diff.json'
REPORT_OUTPUT_FILE_NAME = \
    'legacy_tests_results_report.txt'
SUMMARY_OUTPUT_FILE_NAME = \
    'legacy_tests_results_summary.txt'

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class AliceEvoLegacyTests(sdk2.Task):
    """ Task to launch legacy vins integration tests """

    class Requirements(sdk2.Task.Requirements):
        environments = [sdk_environments.PipEnvironment('startrek_client',
                                                        version='2.3.0',
                                                        custom_parameters=['--upgrade-strategy only-if-needed']), ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60 * 60  # 10 hours

        evo_in_megamind_url_left = sdk2.parameters.String(
            'evo_in_megamind_url_left',
            required=True,
            default=EvoConstants.MEGAMIND_HAMSTER_URL
        )
        evo_in_vins_package_left = sdk2.parameters.Resource(
            "evo_in_vins_package_left",
            resource_type=VinsPackage
        )
        evo_in_megamind_url_right = sdk2.parameters.String(
            'evo_in_megamind_url_right',
            required=True,
            default=EvoConstants.MEGAMIND_HAMSTER_URL
        )
        evo_in_vins_package_right = sdk2.parameters.Resource(
            "evo_in_vins_package_right",
            resource_type=VinsPackage
        )
        evo_in_sandbox_svn_path_right = sdk2.parameters.String(
            'evo_in_sandbox_svn_path_right',
            required=True,
            default=EvoConstants.TRUNK_SANDBOX_SVN_PATH
        )
        evo_in_st_ticket = sdk2.parameters.String(
            'evo_in_st_ticket',
            required=True,
            default=EvoConstants.PLACEHOLDER_ST_TICKET
        )
        with sdk2.parameters.Output:
            evo_out_report_on_gist = sdk2.parameters.String(
                "evo_out_report_on_gist"
            )

    @staticmethod
    def generate_test_cmd(pa_tools_path=None,
                          megamind_url_left=None,
                          megamind_url_right=None,
                          result_filename=None):
        # Example:
        # ./tools/pa/pa_tools run_integration_diff_tests
        # --placeholders "city_dialog_id:672f7477-d3f0-443d-9bd5-2487ab0b6a4c"
        # --fast
        # --trace
        # --experiments "mm_dont_defer_apply:1"
        # --vins-url "http://vins.alice.yandex.net/speechkit/app/pa/"
        # --vins-url-v2 http://alice-megamind-rc-51-7.yappy.beta.yandex.ru/speechkit/app/pa/
        # --json-dump prod_vs_vins_51.json
        # --report prod_vs_vins_51.txt
        return [
            pa_tools_path,
            LegacyConstants.PA_TOOLS_TEST_NAME,
            '--placeholders', LegacyConstants.PLACEHOLDERS,
            '--fast',
            '--trace',
            '--experiments', LegacyConstants.EXPERIMENTS,
            '--vins-url', '"' + megamind_url_left + '"',
            '--vins-url-v2', '"' + megamind_url_right + '"',
            '--json-dump', result_filename,
            '--report', REPORT_OUTPUT_FILE_NAME,
            '--summary', SUMMARY_OUTPUT_FILE_NAME,
        ]

    @staticmethod
    def execute_with_logging(cmd):
        logging.debug('Running {}'.format(cmd))

        s = subprocess.Popen(cmd, stderr=subprocess.PIPE, shell=True, env={'PYTHONIOENCODING': 'utf-8'})
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug("pid %s: %s", os.getpid(), line.strip())

        logging.debug("pid %s: finished", os.getpid(), cmd)
        return exitcode

    def build_pa_tools(self):
        task = sdk2.Task['KOSHER_YA_MAKE'](
            self,
            description='Build pa_tools from Arcadia source',
            arts='alice/vins/tools/pa/pa_tools',
            result_rt='VINS_TOOL_PA_RESOURCE',
            result_single_file=True,
            targets='alice/vins/tools/pa',
            checkout_arcadia_from_url=self.Parameters.evo_in_sandbox_svn_path_right,
        )

        self.Context.pa_tools_task_id = task.id
        task.enqueue()
        return task

    @staticmethod
    def download_resource(tools_resource):
        resource_path = str(sdk2.ResourceData(tools_resource).path)
        pkg_name = os.path.basename(resource_path)
        dst_path = pj(os.getcwd(), pkg_name)
        if os.path.exists(dst_path):
            shutil.rmtree(dst_path)
        logging.info('Copying {} build...'.format(pkg_name))
        shutil.copyfile(resource_path, dst_path)
        return dst_path, pkg_name

    def on_execute(self):
        with self.memoize_stage.run_ya_make:
            subtasks = []
            subtasks.append(self.build_pa_tools())
            raise sdk2.WaitTask(subtasks,
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.check_ya_make_ok:
            subtasks = list(self.find())
            if subtasks and \
               any(task.status != ctt.Status.SUCCESS for task in subtasks):
                raise common_errors.TaskFailure('Build tasks failed!')

            pa_tools_task_id = self.Context.pa_tools_task_id
            pa_tools = sdk2.Resource.find(type=VinsToolPaResource,
                                          task_id=pa_tools_task_id).first()

            pa_tools_path, pa_tools_name = self.download_resource(pa_tools)

            logging.debug('Chmoding {}'.format(pa_tools_name))
            self.execute_with_logging('chmod +x {}'.format(pa_tools_path))

            test_cmd_hamster = \
                ' '.join(self.generate_test_cmd(pa_tools_path,
                                                self.Parameters.evo_in_megamind_url_left,
                                                self.Parameters.evo_in_megamind_url_right,
                                                JSON_OUTPUT_FILE_NAME))

            logging.debug('Execute test_cmd_hamster')
            self.execute_with_logging(test_cmd_hamster)

            logging.debug('START. Upload to gist: ' + REPORT_OUTPUT_FILE_NAME)
            path_to_src = os.path.abspath(REPORT_OUTPUT_FILE_NAME)

            gist_link = None

            with open(path_to_src) as x:
                f = x.read()
                gist_link = GistToolbox.post_diff(REPORT_OUTPUT_FILE_NAME, f)

            logging.debug('DONE. Upload to gist: ' + REPORT_OUTPUT_FILE_NAME)

            self.Parameters.evo_out_report_on_gist = \
                gist_link

            target_ticket_st = self.Parameters.evo_in_st_ticket

            from startrek_client import Startrek
            st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                                 useragent=AccessTokens.ST_USERAGENT)
            StartrekToolbox.append_startrek_comment(
                st_client,
                RunHelper.generate_comment(
                    self.Parameters.evo_in_megamind_url_left,
                    self.Parameters.evo_in_vins_package_left.id,
                    self.Parameters.evo_in_vins_package_right.id,
                    gist_link
                ),
                target_ticket_st,
                LegacyConstants.ST_COMMENT_TITLE,
                False
            )


class RunHelper(object):
    @staticmethod
    def generate_comment(megamind_url_left,
                         vins_package_left_resource_id,
                         vins_package_right_resource_id,
                         gist_link):
        """
        Generate comment
        with cut, summary and link to report

        """

        comment_text = "<{" + \
            MiscToolbox.get_smart_run_id(
                vins_package_left_resource_id,
                vins_package_right_resource_id,
                megamind_url_left == EvoConstants.MEGAMIND_HAMSTER_URL,
                LegacyConstants.LEGACY_DESCRIPTION
            ) + "\n"

        path_to_src = os.path.abspath(SUMMARY_OUTPUT_FILE_NAME)

        with open(path_to_src) as x:
            f = x.read()
            comment_text = comment_text + "%%\n" + f + "\n%%\n"

        comment_text = comment_text + "Full report: " + gist_link + "\n}>\n"

        return comment_text
