import os
import shutil
import json
import logging

from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess as sp
import sandbox.common.errors as ce
import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as sdk_environments

from sandbox.projects.alice_evo.common.const import AccessTokens, EvoConstants
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox
from sandbox.projects.alice_evo.common.sandbox import SandboxToolbox
from sandbox.projects.alice_evo.common.const import RequirementsConstants

VINS_RESOURCES_PATH = 'alice/vins/resources'
NLU_TOOLS_BINARY_PATH = 'alice/vins/tools/nlu/nlu_tools'
VALIDATION_SETS_PATH = 'alice/vins/apps/personal_assistant/personal_assistant/tests/validation_sets'
EXPERIMENTS = 'disable_quasar_forbidden_intents,video_play,how_much,avia,translate,weather_precipitation,tv,tv_stream,quasar_tv,radio_play_in_quasar'

STABLE = 'stable'
RELEASE = 'release'
NON_EXISTING_DIR = 'non_existing_dir'
ERROR_STATS = 'error_stats'
REPORT_FILE_NAME = 'report.txt'

ST_COMMENT_TITLE = 'Alice EVO quality measuring'

logger = logging.getLogger(__name__)


class MeasureQualityReportResource(sdk2.Resource):
    """ Measure quality report resource """


class MeasureQualityValidationSetsResource(sdk2.Resource):
    """ Measure quality validation sets resource """


class MeasureQualityTolokaIntentRenamesResource(sdk2.Resource):
    """ Measure quality toloka intent renames resource """


class AliceEvoMeasureQuality(sdk2.Task):
    """
    Measuring quality in release machine
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [
            sdk_environments.PipEnvironment(
                'startrek_client',
                version='2.3.0',
                custom_parameters=['--upgrade-strategy only-if-needed']
            ),
        ]

        client_tags = RequirementsConstants.CLIENT_TAGS
        cores = RequirementsConstants.CORES
        dns = RequirementsConstants.DNS
        disk_space = RequirementsConstants.DISK_SPACE
        ram = RequirementsConstants.RAM
        privileged = RequirementsConstants.PRIVILEGED

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60 * 60  # 10 hours

        evo_in_stable_svn_path = sdk2.parameters.String(
            'evo_in_stable_svn_path',
            required=True
        )
        evo_in_release_svn_path = sdk2.parameters.String(
            'evo_in_release_svn_path',
            required=True
        )

        validation_sets = sdk2.parameters.Resource(
            'validation_sets',
            resource_type=MeasureQualityValidationSetsResource,
        )
        toloka_intent_renames = sdk2.parameters.Resource(
            'toloka intent renames',
            resource_type=MeasureQualityTolokaIntentRenamesResource,
        )

        evo_in_release_ticket = sdk2.parameters.String(
            'evo_in_release_ticket',
            required=True,
            default=EvoConstants.PLACEHOLDER_ST_TICKET
        )

        evo_in_tag_num = sdk2.parameters.String(
            'evo_in_tag_num',
            required=True,
            default="1"
        )

        stable_build_task = sdk2.parameters.Task(
            'stable_build_task',
            task_type='YA_MAKE'
        )

        release_build_task = sdk2.parameters.Task(
            'release_build_task',
            task_type='YA_MAKE'
        )

        release_result = sdk2.parameters.Resource(
            'release_result'
        )

        stable_pkl_dir = sdk2.parameters.Resource(
            'stable_pkl_dir'
        )

        release_pkl_dir = sdk2.parameters.Resource(
            'release_pkl_dir'
        )

    class Context(sdk2.Task.Context):
        stable_build_subtask_id = None
        release_build_subtask_id = None
        compare_base_summary = None
        launch_calculation = None

    def _get_build_output_path(self, build_subtask_id):
        logger.info('Synchronizing build_output data on disk for task with id={}'.format(build_subtask_id))
        build_subtask = sdk2.Task.find(id=build_subtask_id).first()
        logging.info('Found task with id={}'.format(build_subtask.id))

        build_output = sdk2.Resource['BUILD_OUTPUT'].find(task=build_subtask).first()
        logging.info('Found resource with id={}'.format(build_output.id))
        build_output_data = sdk2.ResourceData(build_output)
        return str(build_output_data.path)

    def _run_process_nlu_on_dataset(self, nlu_tools_binary, env, test_file, output_file, argv_postfix, out, err):
        common_argv_prefix = [
            nlu_tools_binary,
            'process_nlu_on_dataset',
            '--log-level', 'ERROR',
            'personal_assistant',
            'classify',
        ]

        argv = common_argv_prefix \
            + [test_file, '-o', output_file] \
            + argv_postfix

        sp.check_call(argv, env=env, stdout=out, stderr=err)

    def _process_nlu_on_validation_sets(
        self,
        build_output_path,
        validation_sets_dir,
        output_dir,
        out=None,
        err=None
    ):
        logging.info('Starting to process nlu on validation_sets for {}'.format(output_dir))
        if os.path.exists(output_dir):
            os.remove(output_dir)
        os.mkdir(output_dir)

        nlu_tools_binary = os.path.join(build_output_path, NLU_TOOLS_BINARY_PATH)
        env = {
            'VINS_RESOURCES_PATH': os.path.join(build_output_path, VINS_RESOURCES_PATH),
            'VINS_NUM_PROCS': '16',
            'TMP': os.getenv('TMP'),
        }

        test_name = 'quasar_lsr'
        self._run_process_nlu_on_dataset(
            nlu_tools_binary,
            env,
            os.path.join(validation_sets_dir, '{}.tsv'.format(test_name)),
            os.path.join(output_dir, '{}.pkl'.format(test_name)),
            [
                '--app-info', '{"app_id":"ru.yandex.quasar.services"}',
                '--experiments', EXPERIMENTS,
                '--text-col', 'text',
                '--intent-col', 'intent',
                '--prev-intent-col', 'prev_intent',
                '--device-state-col', 'device_state',
                '--apply-item-selection'
            ],
            out,
            err
        )

        test_name = 'search_release'
        self._run_process_nlu_on_dataset(
            nlu_tools_binary,
            env,
            os.path.join(validation_sets_dir, '{}.tsv'.format(test_name)),
            os.path.join(output_dir, '{}.pkl'.format(test_name)),
            [
                '--prev-intent-col', 'prev_intent',
                '--app-info', '{"app_id": "winsearchbar", "platform": "windows"}',
                '--experiments', EXPERIMENTS
            ],
            out,
            err
        )

        test_name = 'auto_release'
        self._run_process_nlu_on_dataset(
            nlu_tools_binary,
            env,
            os.path.join(validation_sets_dir, '{}.tsv'.format(test_name)),
            os.path.join(output_dir, '{}.pkl'.format(test_name)),
            ['--app-info', '{"app_id": "yandex.auto", "platform": "android"}'],
            out,
            err
        )

        test_name = 'navi_release'
        self._run_process_nlu_on_dataset(
            nlu_tools_binary,
            env,
            os.path.join(validation_sets_dir, '{}.tsv'.format(test_name)),
            os.path.join(output_dir, '{}.pkl'.format(test_name)),
            ['--app-info', '{"app_id":"ru.yandex.yandexnavi"}'],
            out,
            err
        )

        logging.info('Result files: {}'.format(os.listdir(output_dir)))

    def _compare_base(
        self,
        release_nlu_tools_binary,
        baseline_dir,
        results_dir,
        toloka_intent_renames_dir,
        out=None,
        err=None
    ):
        logging.info('Measure search-or-converse on the Search release dataset')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'search_release.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames_search_converse.json'),
            '--errors',
            '--baseline', os.path.join(baseline_dir, 'mobile-search-or-converse.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)
        os.rename(
            os.path.join(results_dir, 'search_release.intent.errors'),
            os.path.join(results_dir, 'mobile-search-or-converse.intent.errors')
        )
        os.rename(
            os.path.join(results_dir, 'search_release.intent.metrics.json'),
            os.path.join(results_dir, 'mobile-search-or-converse.intent.metrics.json')
        )

        logging.info('Measure search-or-converse on the Quasar release dataset')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'quasar_lsr.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames_search_converse.json'),
            '--errors',
            '--baseline', os.path.join(baseline_dir, 'quasar-search-or-converse.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)
        os.rename(
            os.path.join(results_dir, 'quasar_lsr.intent.errors'),
            os.path.join(results_dir, 'quasar-search-or-converse.intent.errors')
        )
        os.rename(
            os.path.join(results_dir, 'quasar_lsr.intent.metrics.json'),
            os.path.join(results_dir, 'quasar-search-or-converse.intent.metrics.json')
        )

        logging.info('Measure Search release dataset')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'search_release.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames.json'),
            '--errors',
            '--other', '.*other',
            '--baseline', os.path.join(baseline_dir, 'search_release.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)

        logging.info('Measure Auto release dataset')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'auto_release.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames_auto.json'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames.json'),
            '--errors',
            '--other', '.*other',
            '--baseline', os.path.join(baseline_dir, 'auto_release.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)

        logging.info('Measure Navi release datase')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'navi_release.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames_auto.json'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames.json'),
            '--errors',
            '--other', '.*other',
            '--baseline', os.path.join(baseline_dir, 'navi_release.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)

        logging.info('Measure quasar LSR dataset')
        argv = [
            release_nlu_tools_binary,
            'process_nlu_on_dataset', 'personal_assistant', 'report',
            os.path.join(results_dir, 'quasar_lsr.pkl'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames_quasar.json'),
            '--rename', os.path.join(toloka_intent_renames_dir, 'toloka_intent_renames.json'),
            '--errors',
            '--other', '.*other',
            '--baseline', os.path.join(baseline_dir, 'quasar_lsr.intent.metrics.json')
        ]
        sp.check_call(argv, stdout=out, stderr=err)

        if os.path.exists(ERROR_STATS):
            os.remove(ERROR_STATS)

        logging.info('Compare p-values with critical values')
        argv = [
            release_nlu_tools_binary,
            'compare_reports', 'compare_p_values',
            os.path.join(results_dir, 'quasar_lsr.intent.metrics.json'),
            '-s', '30',
            '-p', '0.01',
            '--error-stats', ERROR_STATS
        ]
        sp.check_call(argv, stdout=out, stderr=err)

        argv = [
            release_nlu_tools_binary,
            'compare_reports', 'report_and_exit',
            ERROR_STATS
        ]
        sp.call(argv, stdout=out, stderr=err)

    def _save_compare_base_summary(self):
        expected_metric_files = {
            'auto_release.intent.metrics.json',
            'mobile-search-or-converse.intent.metrics.json',
            'navi_release.intent.metrics.json',
            'quasar-search-or-converse.intent.metrics.json',
            'quasar_lsr.intent.metrics.json',
            'search_release.intent.metrics.json',
        }

        metric_files = {filename for filename in os.listdir('release') if filename.endswith('intent.metrics.json')}

        summary = '**Compare base summary**:\n'
        summary += '#|\n'

        if metric_files == expected_metric_files:
            summary += '|| All **{}** metric files are presented in report: {} | !!(green)OK!! ||\n' \
                .format(len(expected_metric_files), ', '.join(expected_metric_files))
        else:
            summary += '|| Presented metric files ({}) aren\'t expected | !!(red)FAIL!! ||\n' \
                .format(metric_files)
            print(expected_metric_files - metric_files)

        with open(os.path.join(RELEASE, 'quasar_lsr.intent.metrics.json'), 'r') as quasar_lsr_metrics_file:
            quasar_lsr_metrics = json.load(quasar_lsr_metrics_file)

        pa_music_play_report = quasar_lsr_metrics['report']['personal_assistant.scenarios.music_play']
        support = pa_music_play_report.get('support')
        predictions = pa_music_play_report.get('predictions')

        if support is None or predictions is None:
            summary += '|| There is no "support" or "predictions"' \
                       ' field in personal_assistant.scenarios.music_play report' \
                       ' | !!(red)FAIL!! ||\n'
        else:
            percentage = 100.0 * predictions / support
            summary += '|| **{:.2f}%** correctly predicted samples inpersonal_assistant.scenarios.music_play ({}/{}) |' \
                .format(percentage, predictions, support)
            summary += '!!(green)OK!!' if percentage > 90.0 else '!!(red)FAIL!!'
            summary += '||\n'

        if os.path.exists(ERROR_STATS):
            summary += '|| {} are present | !!(red)FAIL!! ||\n'.format(ERROR_STATS)
        else:
            summary += '|| {} are not present | Everything is !!(green)OK!! ||\n'.format(ERROR_STATS)

        summary += '|#'

        self.Context.compare_base_summary = summary

    def on_execute(self):
        logger.info('Starting task')

        self.Context.launch_calculation = self.Parameters.evo_in_tag_num == "1" or \
            self.Parameters.release_result is not None

        if not self.Context.launch_calculation:
            return

        if self.Parameters.release_result is not None:
            release_result = str(sdk2.ResourceData(self.Parameters.release_result).path)
            shutil.copytree(release_result, RELEASE)
            self._save_compare_base_summary()
            return

        if self.Parameters.stable_build_task is not None and self.Parameters.release_build_task is not None:
            self.Context.stable_build_subtask_id = self.Parameters.stable_build_task.id
            self.Context.release_build_subtask_id = self.Parameters.release_build_task.id
        else:
            with self.memoize_stage.run_build_tasks:
                subtasks = []

                stable_build_subtask = sdk2.Task['KOSHER_YA_MAKE'](
                    self,
                    checkout_arcadia_from_url=self.Parameters.evo_in_stable_svn_path,
                    targets=';'.join([VINS_RESOURCES_PATH, NLU_TOOLS_BINARY_PATH]),
                    kill_timeout=5*60*60
                ).enqueue()
                subtasks.append(stable_build_subtask)
                self.Context.stable_build_subtask_id = stable_build_subtask.id

                release_build_subtask = sdk2.Task['KOSHER_YA_MAKE'](
                    self,
                    checkout_arcadia_from_url=self.Parameters.evo_in_release_svn_path,
                    targets=';'.join([VINS_RESOURCES_PATH, NLU_TOOLS_BINARY_PATH]),
                    kill_timeout=5*60*60
                ).enqueue()
                subtasks.append(release_build_subtask)
                self.Context.release_build_subtask_id = release_build_subtask.id

                raise sdk2.WaitTask(
                    subtasks,
                    ctt.Status.Group.FINISH | ctt.Status.Group.BREAK,
                    wait_all=True
                )

        stable_build_output_path = self._get_build_output_path(self.Context.stable_build_subtask_id)
        release_build_output_path = self._get_build_output_path(self.Context.release_build_subtask_id)

        validation_sets_resource = self.Parameters.validation_sets or \
            MeasureQualityValidationSetsResource.find().order(-sdk2.Resource.id).first()
        if validation_sets_resource is None:
            raise ce.TaskError('Validation sets resource was not found')

        toloka_intent_renames_resource = self.Parameters.toloka_intent_renames or \
            MeasureQualityTolokaIntentRenamesResource.find().order(-sdk2.Resource.id).first()
        if toloka_intent_renames_resource is None:
            raise ce.TaskError('Toloka intent renames resource was not found')

        validation_sets_dir = str(sdk2.ResourceData(validation_sets_resource).path)
        logging.info('Validation sets directory: {}'.format(validation_sets_dir))

        release_nlu_tools_binary = os.path.join(release_build_output_path, NLU_TOOLS_BINARY_PATH)
        toloka_intent_renames_dir = str(sdk2.ResourceData(toloka_intent_renames_resource).path)
        logging.info('Toloka intent renames directory: {}'.format(toloka_intent_renames_dir))

        if self.Parameters.stable_pkl_dir is not None and self.Parameters.release_pkl_dir is not None:
            stable_pkl_dir = str(sdk2.ResourceData(self.Parameters.stable_pkl_dir).path)
            shutil.copytree(stable_pkl_dir, STABLE)

            release_pkl_dir = str(sdk2.ResourceData(self.Parameters.release_pkl_dir).path)
            shutil.copytree(release_pkl_dir, RELEASE)
        else:
            with sdk2.helpers.ProcessLog(self, logger="stable_process_nlu") as pl:
                self._process_nlu_on_validation_sets(
                    stable_build_output_path,
                    validation_sets_dir,
                    STABLE,
                    pl.stdout,
                    sp.STDOUT
                )

            with sdk2.helpers.ProcessLog(self, logger="release_process_nlu") as pl:
                self._process_nlu_on_validation_sets(
                    release_build_output_path,
                    validation_sets_dir,
                    RELEASE,
                    pl.stdout,
                    sp.STDOUT
                )

        if os.path.exists(NON_EXISTING_DIR):
            os.remove(NON_EXISTING_DIR)

        with sdk2.helpers.ProcessLog(self, logger="compare_base_non_exist_stable") as pl:
            self._compare_base(
                release_nlu_tools_binary,
                NON_EXISTING_DIR,
                STABLE,
                toloka_intent_renames_dir,
                pl.stdout,
                sp.STDOUT
            )

        with sdk2.helpers.ProcessLog(self, logger="compare_base_stable_release") as pl:
            self._compare_base(
                release_nlu_tools_binary,
                STABLE,
                RELEASE,
                toloka_intent_renames_dir,
                pl.stdout,
                sp.STDOUT
            )

        self._save_compare_base_summary()

    def on_finish(self, prev_status, status):
        logger.info('Finishing task')

        if self.Parameters.release_result is not None:
            release_result = str(sdk2.ResourceData(self.Parameters.release_result).path)
            if not os.path.exists(RELEASE):
                shutil.copytree(release_result, RELEASE)
            self._save_compare_base_summary()

            logger.info('Creating report file')
            report_resource = MeasureQualityReportResource(self, 'Report file', REPORT_FILE_NAME)
            report_data = sdk2.ResourceData(report_resource)

            shutil.copy(os.path.join(release_result, REPORT_FILE_NAME), str(report_data.path))
        elif self.Context.launch_calculation:
            log_resources_count = sdk2.Resource.find(task=self, type='TASK_LOGS').count
            logger.info('Log resouces count: {}'.format(log_resources_count))

            task_logs_resource = sdk2.Resource.find(task=self, type='TASK_LOGS').offset(log_resources_count - 2).first()
            task_logs_data = sdk2.ResourceData(task_logs_resource)

            logger.info('Creating report file')
            report_resource = MeasureQualityReportResource(self, 'Report file', REPORT_FILE_NAME)
            report_data = sdk2.ResourceData(report_resource)

            compare_base_stable_release_path = task_logs_data.path.joinpath('compare_base_stable_release.out.log')
            if compare_base_stable_release_path.exists():
                shutil.copy(str(compare_base_stable_release_path), str(report_data.path))

        if self.Context.launch_calculation:
            from startrek_client import Startrek
            st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                                 useragent=AccessTokens.ST_USERAGENT)
            StartrekToolbox.append_startrek_comment(
                st_client,
                RunHelper.generate_comment(
                    SandboxToolbox.get_sandbox_task_url(str(self.id)),
                    SandboxToolbox.get_sandbox_resource_url(str(report_resource.id)),
                    self.Context.compare_base_summary
                ),
                self.Parameters.evo_in_release_ticket,
                ST_COMMENT_TITLE,
                st_comment_need_trunk_modified=False
            )


class RunHelper(object):
    @staticmethod
    def generate_comment(sandbox_task,
                         sandbox_report_resource,
                         compare_base_summary):
        """
        Generate comment with report
        """
        comment_text = '{}\n'.format(sandbox_task)
        comment_text += 'Report: {}\n'.format(sandbox_report_resource)
        comment_text += compare_base_summary

        return comment_text
