# -*- coding: utf-8 -*-

import json
import logging
import os
import shutil
import time
from os.path import join

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as sdk_environments
from sandbox import common
from sandbox import sdk2
from sandbox.projects.common import network
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.paths import get_logs_folder

from sandbox.projects.alice_evo.AliceEvoIntegrationTestsWrapper import AliceEvoIntegrationTestsWrapper
from sandbox.projects.alice_evo.common.const import AccessTokens, RequirementsConstants
from sandbox.projects.vins.common.resources import VinsPackage, MegamindShooterBinary

logger = logging.getLogger(__name__)


DEFAULT_CONFIG = {
    'joker_config': {
        'server_type': 'plain',
        'server': {
            'host': 'joker-light.alice.yandex.net',
            'port': 80,
        },
        'session_settings': 'fetch_if_not_exists=1&dont_save=1',
        'session_id': 'evo_tests',
    },

    'yav_secret_id': 'sec-01cnbk6vvm6mfrhdyzamjhm4cm',

    'servers_settings': {
        'bass_threads': 20,
        'vins_workers': 10,
    },

    'runs_settings': [
        {
            'config': 'rc',
            'increase_timeouts': False,
            'dont_close': True,
        },
    ],

    'requests_limit': 0,
}


class AliceEvoOnVinsPackage(sdk2.Task):
    """
        Task to run AliceIntegrationTests on VINS_PACKAGE
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [sdk_environments.PipEnvironment('PyYAML')]
        client_tags = RequirementsConstants.CLIENT_TAGS
        cores = RequirementsConstants.CORES
        dns = RequirementsConstants.DNS
        disk_space = RequirementsConstants.DISK_SPACE
        ram = RequirementsConstants.RAM
        privileged = RequirementsConstants.PRIVILEGED

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Packages') as packages_parameters:
            vins_package = sdk2.parameters.Resource('VINS package', resource_type=VinsPackage, required=True)
            shooter_binary = sdk2.parameters.Resource('Megamind shooter resource', resource_type=MegamindShooterBinary, required=True)

        with sdk2.parameters.Group('Settings') as config_parameters:
            config = sdk2.parameters.JSON('Shooter config to use', default=DEFAULT_CONFIG)
            joker_semaphore_name = sdk2.parameters.String('Joker semaphore name')

        kill_timeout = 5 * 60 * 60  # 5 hours

    class Context(sdk2.Task.Context):
        subtask_id = None

    def on_enqueue(self):
        if self.Parameters.joker_semaphore_name:
            self.Requirements.semaphores = ctt.Semaphores(
                acquires=[
                    ctt.Semaphores.Acquire(name=self.Parameters.joker_semaphore_name),
                ],
                release=(ctt.Status.Group.BREAK, ctt.Status.Group.FINISH),
            )

    def prepare_vins_package(self, vins_pkg_path):
        pkg_name = os.path.basename(vins_pkg_path)
        dst_path = join(os.getcwd(), pkg_name)
        if os.path.exists(dst_path):
            shutil.rmtree(dst_path)
        logger.info('Copying VINS build...')
        shutil.copytree(vins_pkg_path, dst_path)
        logger.info('Copying VINS build ended')
        return dst_path

    def on_execute(self):
        yav_token = str(sdk2.Vault.data(AccessTokens.BASS_VAULT_TOKEN_OWNER, AccessTokens.BASS_VAULT_TOKEN_NAME))

        # load packages
        vins_package_dir = self.prepare_vins_package(str(sdk2.ResourceData(self.Parameters.vins_package).path))
        shooter_binary = str(sdk2.ResourceData(self.Parameters.shooter_binary).path)

        sp.Popen(['chmod', '-R', '755', vins_package_dir]).wait()

        # write config
        run_settings = self.Parameters.config['runs_settings'][0]
        run_settings['package_path'] = vins_package_dir
        run_settings['logs_path'] = get_logs_folder()

        config_path = join(get_logs_folder(), "config.json")
        with open(config_path, 'w') as cf:
            json.dump(self.Parameters.config, cf)

        # launch shooter
        out_path = join(get_logs_folder(), "shooter.out")
        err_path = join(get_logs_folder(), "shooter.err")

        with open(out_path, 'w') as out, open(err_path, 'w') as err:
            sp.Popen(
                [
                    shooter_binary, 'run', config_path,
                    '--yav-token', yav_token,
                ],
                cwd=get_logs_folder(),
                stdout=out,
                stderr=err,
                stdin=None,
            ).wait()

        # get Megamind port
        with open(join(get_logs_folder(), 'ports.json'), 'r') as f:
            megamind_port = json.load(f)['megamind']

        # launch subtask
        logger.info('Sandbox IPv6: {}'.format(network.get_my_ipv6()))

        if self.Context.arcadia_patch is ctm.NotExists:
            self.Context.arcadia_patch = ''
        subtask = AliceEvoIntegrationTestsWrapper(
            self,
            owner=self.Parameters.owner,
            arcadia_patch=self.Context.arcadia_patch,
            megamind_url='http://[{}]:{}/speechkit/app/pa/'.format(network.get_my_ipv6(), megamind_port),
            uniproxy_url='wss://beta.uniproxy.alice.yandex.net/alice-uniproxy-hamster/uni.ws'
        ).enqueue()
        self.Context.subtask_id = subtask.id

        while self.find(AliceEvoIntegrationTestsWrapper, status=list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK)).count == 0:
            logger.info('No ended subtasks, sleeping...')
            time.sleep(60)

        subtask = self.find(id=self.Context.subtask_id).first()
        if subtask.status not in ctt.Status.Group.SUCCEED:
            raise common.errors.TaskFailure('Some tests failed, check them please')

    @sdk2.header()
    def head(self):
        return 'Full test report: <a href="{href}">{href}</a>'.format(href=common.utils.get_task_link(self.Context.subtask_id) + '/report')
