
# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.projects.alice_evo.common.const import (AccessTokens,
                                                     EvoConstants,
                                                     PerfConstants)
from sandbox.projects.alice_evo.common.misc import MiscToolbox
from sandbox.projects.alice_evo.common.sandbox import SandboxToolbox
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox
from sandbox.projects.vins.common.resources import (VinsPackage,
                                                    MegamindHelperBinaries,
                                                    MegamindRequests,
                                                    MegamindShooterBinary)
import sandbox.common.types.task as ctt
import sandbox.sandboxsdk.environments as sdk_environments

logger = logging.getLogger(__name__)

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]


class AliceEvoPerfWrapper(sdk2.Task):
    """
        Handle perf test launch for BMV
    """

    class Requirements(sdk2.Task.Requirements):
        environments = [sdk_environments.PipEnvironment('startrek_client',
                                                        version='2.3.0',
                                                        custom_parameters=['--upgrade-strategy only-if-needed']), ]

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min

        evo_in_vins_package_left = sdk2.parameters.Resource(
            "evo_in_vins_package_left",
            resource_type=VinsPackage
        )
        evo_in_vins_package_right = sdk2.parameters.Resource(
            "evo_in_vins_package_right",
            resource_type=VinsPackage
        )
        evo_in_is_left_package_prod = sdk2.parameters.Bool(
            'evo_in_is_left_package_prod',
            required=True,
            default=True
        )
        evo_in_is_right_package_from_trunk = sdk2.parameters.Bool(
            "evo_in_is_right_package_from_trunk",
            required=True,
            default=False
        )
        evo_in_target_ticket = sdk2.parameters.String(
            "evo_in_target_ticket",
            required=True,
            default=EvoConstants.PLACEHOLDER_ST_TICKET
        )
        evo_in_use_dummy_perf_task = sdk2.parameters.Bool(
            "evo_in_use_dummy_perf_task",
            required=True,
            default=False
        )

        with sdk2.parameters.Output:
            evo_out_launched_perf_task = sdk2.parameters.String(
                "evo_out_launched_perf_task"
            )

    def on_execute(self):
        logger.info("[INFO] Started on_execute")

        logger.info("[INFO] Before task creation")
        old_package = self.Parameters.evo_in_vins_package_left
        new_package = self.Parameters.evo_in_vins_package_right

        smart_run_id = \
            MiscToolbox.get_smart_run_id(
                old_package.id,
                new_package.id,
                self.Parameters.evo_in_is_left_package_prod,
                PerfConstants.PERF_DESCRIPTION
            )

        target_task = PerfConstants.REAL_TASK
        if self.Parameters.evo_in_use_dummy_perf_task:
            target_task = PerfConstants.DUMMY_TASK

        with self.memoize_stage.create_children:
            task = sdk2.Task[target_task](
                self,

                old_vins_package=old_package,
                new_vins_package=new_package,
                megamind_requests=sdk2.Resource.find(
                    type=MegamindRequests,
                    id=PerfConstants.MEGAMIND_REQUESTS).first(),
                shooter_binary=sdk2.Resource.find(
                    type=MegamindShooterBinary,
                    id=PerfConstants.SHOOTER_BINARY).first(),
                perf_differ_binary=sdk2.Resource.find(
                    type=MegamindHelperBinaries,
                    id=PerfConstants.PERF_DIFFER_BINARY).first(),

                joker_semaphore_name=PerfConstants.JOKER_SEMAPHORE_NAME,

                description=smart_run_id,
                kill_timeout=2 * 60 * 60
            )
            task.enqueue()
            self.Context.perf_task_id = task.id
            raise sdk2.WaitTask([self.Context.perf_task_id], WAIT_STATUSES, wait_all=True)

        out_resource = sdk2.Resource.find(type=PerfConstants.OUTPUT_RESOURCE, task_id=self.Context.perf_task_id).first()

        logger.info("[INFO] After task creation")
        self.Parameters.evo_out_launched_perf_task = \
            SandboxToolbox.get_sandbox_task_url(str(self.Context.perf_task_id))

        target_ticket_st = self.Parameters.evo_in_target_ticket

        from startrek_client import Startrek
        st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                             useragent=AccessTokens.ST_USERAGENT)
        StartrekToolbox.append_startrek_comment(
            st_client,
            RunHelper.get_comment_text(
                self.Parameters.evo_out_launched_perf_task,
                SandboxToolbox.get_sandbox_resource_url(str(out_resource.id)),
            ),
            target_ticket_st,
            PerfConstants.ST_COMMENT_TITLE,
            self.Parameters.evo_in_is_right_package_from_trunk,
        )

        logger.info("[INFO] Task id: " + str(self.Context.perf_task_id))


class RunHelper(object):
    @staticmethod
    def get_comment_text(sandbox_perf_task, sandbox_perf_resource):
        """
        Get comment text for Startrek

        """

        text = sandbox_perf_task + "\n"
        text += "<{Графики и данные перф-теста" + "\n"
        text += "**((" + sandbox_perf_resource + "/perc.txt" + " ПЕРЦЕНТИЛИ ТУТ))**" + "\n"
        text += sandbox_perf_resource + "/hist.png" + "\n"
        text += sandbox_perf_resource + "/sorted.png" + "\n"
        text += "}>" + "\n"

        return text
