import logging
import json

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types.notification import Transport

logger = logging.getLogger(__name__)


class AliceEvoResultsDiff(sdk2.Task):

    class Requirements(sdk2.Requirements):
        # tasks_resource = sdk2.Task.Requirements.tasks_resource(default=2371915401)
        pass

    class Parameters(sdk2.Parameters):
        kill_timeout = 3600

        with sdk2.parameters.Group('ARCANUM_TOKEN from Sandbox Vault') as yav_block:
            arcanum_token_name = sdk2.parameters.String('Name', required=True, default='robot-bassist_arcanum_token')
            arcanum_token_owner = sdk2.parameters.String('Owner', required=True, default='robot-bassist')

        baseline_task_id = sdk2.parameters.Integer('baseline_task_id', required=True)
        test_task_id = sdk2.parameters.Integer('test_task_id', required=True)
        only_release = sdk2.parameters.Bool('only_release', default=True)
        report_to_mail = sdk2.parameters.Bool(
            "Send error mails",
            required=False,
            default=False,
        )
        with report_to_mail.value[True]:
            urgent = sdk2.parameters.Bool(
                "Urgent emails",
                required=True,
                default=False,
            )
            mail_recepients = sdk2.parameters.List(
                label="Recepients of error messages",
                value_type=sdk2.parameters.Staff,
                required=True
            )

    def _get_response(self, task_id):
        from alice.tests.evo_parser.lib import EvoFailsParser
        from alice.tests.evo_parser.lib.evo_parser_data_pb2 import TEvoParserRequest

        token = sdk2.Vault.data(self.Parameters.arcanum_token_owner, self.Parameters.arcanum_token_name)
        return EvoFailsParser(TEvoParserRequest(
            ArcanumToken=token,
            SandboxTaskId=task_id,
            OnlyRelease=self.Parameters.only_release,
        )).parse()

    def _build_failed_set(self, response):
        import google.protobuf.text_format as text
        failed_set = set()
        for test_pack in response.FailedTestsPacks:
            for test in test_pack.Tests:
                failed_set.add(text.MessageToString(test, as_utf8=True))

        return failed_set

    def _build_diff(self, baseline_response, test_response):
        failed_baseline = self._build_failed_set(baseline_response)
        failed_test = self._build_failed_set(test_response)

        new_failed = failed_test.difference(failed_baseline)
        new_fixed = failed_baseline.difference(failed_test)

        return new_failed, new_fixed

    def on_execute(self):
        from google.protobuf.json_format import MessageToDict

        baseline_response = self._get_response(self.Parameters.baseline_task_id)
        logger.info('EvoFailedParser result for baseline:\n' + json.dumps(MessageToDict(baseline_response), indent=4))

        test_response = self._get_response(self.Parameters.test_task_id)
        logger.info('EvoFailedParser result for test:\n' + json.dumps(MessageToDict(test_response), indent=4))

        failed, fixed = self._build_diff(baseline_response, test_response)

        report = ['Failed tests: {}'.format(len(failed))]
        for failed_test in failed:
            report.append(failed_test)

        report.append('Fixed tests: {}'.format(len(fixed)))
        for fixed_test in fixed:
            report.append(fixed_test)

        self.set_info('<br>'.join(report), do_escape=False)

        logger.info('\nFailed {}: {}\nFixed {}: {}'.format(len(failed), failed, len(fixed), fixed))

        if failed:
            if self.Parameters.report_to_mail and self.Parameters.mail_recepients:
                logging.debug("Send emails to {}".format(self.Parameters.mail_recepients))
                mail_body = ['Error in task https://sandbox.yandex-team.ru/task/{}'.format(self.id)] + report
                self.server.notification(
                    subject="Alice EVO Diff on CI has failed, task {}".format(self.id),
                    body="\n".join(mail_body),
                    recipients=self.Parameters.mail_recepients,
                    transport=Transport.EMAIL,
                    urgent=self.Parameters.urgent
                )
            raise errors.TaskFailure('New failed tests')
