# -*- coding: utf-8 -*-

import datetime
import json
import logging
import requests


from sandbox import sdk2
from sandbox.common import errors as common_errors
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.alice_evo.common.const import AccessTokens
from sandbox.projects.alice_evo.common.vault import VaultToolbox


logger = logging.getLogger(__name__)

PREFIX_TESTSUITE = 'https://testpalm.yandex-team.ru/alice/testsuite/'
PREFIX_TESTRUN = 'https://testpalm.yandex-team.ru/alice/testrun/'

STATFACE_PROJECT = 'VoiceTech/leletko/MegamindTestpalmAutomation'
STATFACE_SCALE = 'd'
STATFACE_TIMEOUT = 600  # in seconds
STATFACE_URL = 'https://upload.stat.yandex-team.ru/_api/report/data'

TARGET_PERCENTAGE = 75

TESTPALM_API_URL = 'https://testpalm-api.yandex-team.ru'
TESTPALM_API_TIMEOUT = 600  # in seconds
TESTPALM_PROJECT = 'alice'

TOKENS_ERROR = 'Failed to get tokens from vault!'


class TestpalmConnector(object):
    __api_url = TESTPALM_API_URL
    __api_timeout = TESTPALM_API_TIMEOUT
    __project_id = TESTPALM_PROJECT

    def __init__(self, testpalm_token=None):
        self.__token = testpalm_token

    def __generate_headers(self):
        return {
            'Authorization': 'OAuth {}'.format(self.__token),
            'Content-Type': 'application/json',
        }

    def __make_api_request(self, path, params={}):
        logger.info(
            'Making api request to path: {path}'.format(
                path=path,
            )
        )

        url = '{base_url}{path}'.format(
            base_url=self.__api_url,
            path=path,
        )

        headers = self.__generate_headers()

        response = requests.get(url,
                                headers=headers,
                                params=params,
                                timeout=self.__api_timeout)

        if response.status_code != 200:
            raise Exception(
                'Testpalm response error, code: {code}, body: {body}'.format(
                    code=response.status_code,
                    body=response.text,
                )
            )

        response_json = response.json()

        logger.info(
            'Received response: {response_json}'.format(
                response_json=response_json,
            )
        )

        return response_json

    def get_all_alice_automated_testcases(self):
        path = '/testcases/{project_id}'.format(
            project_id=self.__project_id,
        )

        params = {
            'include': 'id,attributes.5e553bd08954da224a6a723e',
        }

        # Answer example (truncated):
        # https://testpalm-api.yandex-team.ru/testcases/alice?include=id%2Cattributes.5e553bd08954da224a6a723e
        # [
        # {
        #   "attributes": {},
        #   "id": 2050
        # },
        # {
        #   "attributes": {
        #     "5e553bd08954da224a6a723e": [
        #       "https://a.yandex-team.ru/arc/trunk/arcadia/alice/tests/integration_tests/test_alarm.py"
        #     ]
        #   },
        #   "id": 2051
        # },
        # {
        #   "attributes": {},
        #   "id": 2052
        # },
        # ]
        resp = self.__make_api_request(path, params=params)

        ids = {}
        for item in resp:
            if '5e553bd08954da224a6a723e' in item.get('attributes', {}):
                ids[item['id']] = True

        return ids

    def get_megamind_testsuites(self):
        path = '/testsuite/{project_id}'.format(
            project_id=self.__project_id,
        )

        expression = {'type': 'CONTAIN', 'key': 'tags', 'value': 'Megamind'}

        params = {
            'include': 'id',
            'expression': json.dumps(expression),
        }

        resp = self.__make_api_request(path, params=params)

        ids = []
        for item in resp:
            ids.append(item['id'])

        return ids

    def get_testruns(self, testsuite_id=None):
        path = '/testrun/{project_id}'.format(
            project_id=self.__project_id,
        )

        params = {
            'include': 'id,title,createdTime',
            'testSuite._id': testsuite_id,
        }

        unsorted_response = self.__make_api_request(path, params=params)
        resp = sorted(unsorted_response, key=lambda k: k['createdTime'])

        ids = []
        for item in resp:
            if 'ALICERELEASE' not in item['title']:
                continue
            if 'MEGAMIND' not in item['title']:
                continue
            if '31.05' not in item['title']:
                continue
            ids.append(item['id'])

        return ids

    def get_testrun_resolution_and_included_tests(self, testrun_id=None):
        path = '/testrun/{project_id}'.format(
            project_id=self.__project_id,
        )

        params = {
            'include': 'id,resolution,testGroups',
            'id': testrun_id,
        }

        resp = self.__make_api_request(path, params=params)

        # Resolution Details Example
        #
        # 'resolution': {
        #     'counter': {
        #         'passed': 3,
        #         'failed': 0,
        #         'skipped': 0,
        #         'broken': 0,
        #         'created': 0,
        #         'started': 0,
        #         'knownbug': 0,
        #         'blocked': 0,
        #         'total': 3
        #     },
        #     'issues': [],
        #     'summary': 'passed'
        # }

        included_tests = []

        main_item = resp[0]

        for test_group in main_item['testGroups']:
            for test_case in test_group['testCases']:
                test_case_id = \
                    test_case['testCase']['id']
                included_tests.append(test_case_id)

            if test_case["status"] != "PASSED":
                logger.info("FOUND_BROKEN_TESTCASE " + str(test_case_id))

        return main_item['resolution'], \
            included_tests


class StatfaceConnector(object):
    __stat_project = STATFACE_PROJECT
    __stat_scale = STATFACE_SCALE
    __stat_timeout = STATFACE_TIMEOUT
    __stat_url = STATFACE_URL

    def __init__(self, statface_token=None):
        self.__token = statface_token

    def __generate_headers(self):
        return {
            'Authorization': 'OAuth {}'.format(self.__token),
        }

    def __make_api_request(self, data={}):
        logger.info('Making statface api request to path')

        response = requests.post(self.__stat_url,
                                 headers=self.__generate_headers(),
                                 data=data,
                                 timeout=self.__stat_timeout)

        if response.status_code != 200:
            raise Exception(
                'Statface response error, code: {code}, body: {body}'.format(
                    code=response.status_code,
                    body=response.text,
                )
            )

        response_text = str(response.text)

        logger.info(
            'Received response: {response_text}'.format(
                response_text=response_text,
            )
        )

    def update_statface_graph(self, target=0, current=0, total=0):
        date_str = str(datetime.datetime.now().date())
        content = [{'fielddate': date_str,
                    'target': target,
                    'total': total,
                    'current': current}]
        data = {
            'name': self.__stat_project,
            'scale': 'd',
            'data': json.dumps({'values': content}),
        }
        self.__make_api_request(data=data)


def calculate_target_count(testpalm_token):
    target_count = 0
    automated_count = 0

    testpalm_api = TestpalmConnector(testpalm_token)

    all_alice_automated_testcases = \
        testpalm_api.get_all_alice_automated_testcases()

    logger.info('Automated testcases')
    logger.info(json.dumps(all_alice_automated_testcases))

    megamind_testsuites = testpalm_api.get_megamind_testsuites()
    for testsuite_id in megamind_testsuites:
        logger.info('Start testsuite processing')
        logger.info('Testsuite: ' + PREFIX_TESTSUITE + testsuite_id)

        testruns = testpalm_api.get_testruns(testsuite_id=testsuite_id)
        if len(testruns) == 0:
            logger.info('No testruns for testsuite')
            continue
        testrun_id = testruns[-1]
        logger.info('Testrun: ' + PREFIX_TESTRUN + testrun_id)

        resolution, included_tests \
            = testpalm_api.get_testrun_resolution_and_included_tests(
                testrun_id=testrun_id)
        logger.info('Resolution: ' + json.dumps(resolution))

        for test_case_id in included_tests:
            logger.info('Check included testcase ' + str(test_case_id))
            if test_case_id in all_alice_automated_testcases:
                logger.info('Found automated testcase ' + str(test_case_id))
                automated_count += 1
            else:
                logger.info('Requires automation ' + str(test_case_id))

        target_count += resolution['counter']['passed']
        logger.info('Finish testsuite processing')

    return int((target_count * TARGET_PERCENTAGE) / 100.0), \
        target_count, \
        automated_count


def update_statistics(statface_token, target, current, total):
    statface_api = StatfaceConnector(statface_token)

    statface_api.update_statface_graph(target, current, total)


class AliceEvoTestpalmScore(sdk2.Task):
    '''
        Calculate automation level for Megamind Testpalm test cases
    '''

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min

    def on_execute(self):
        testpalm_token = None
        statface_token = None

        try:
            testpalm_token \
                = VaultToolbox.get_token_from_env_or_vault(
                    AccessTokens.TESTPALM_TOKEN_OWNER,
                    AccessTokens.TESTPALM_TOKEN_NAME,
                    AccessTokens.TESTPALM_TOKEN_ENV
                )
            statface_token \
                = VaultToolbox.get_token_from_env_or_vault(
                    AccessTokens.STATFACE_TOKEN_OWNER,
                    AccessTokens.STATFACE_TOKEN_NAME,
                    AccessTokens.STATFACE_TOKEN_ENV
                )

        except Exception as exc:
            eh.log_exception(TOKENS_ERROR, exc)

        if (not testpalm_token) or (not statface_token):
            logger.error(TOKENS_ERROR)
            raise common_errors.TaskFailure(TOKENS_ERROR)

        target, total, current = \
            calculate_target_count(testpalm_token)

        logger.info('STATUS: calculated target: ' + str(target))
        logger.info('STATUS: calculated total: ' + str(total))
        logger.info('STATUS: calculated current: ' + str(current))

        update_statistics(statface_token, target, current, total)
