
# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
from sandbox.projects.alice_evo.common.sandbox import SandboxToolbox
from sandbox.projects.alice_evo.common.yappy import YappyToolbox
from sandbox.projects.vins.common.resources import VinsPackage
from sandbox.common.types import task as ctt
from sandbox.common import errors as common_errors

logger = logging.getLogger(__name__)

WAIT_STATUSES = [
    ctt.Status.Group.FINISH,
    ctt.Status.Group.BREAK
]

BETA_CONSISTENCY_TIMEOUT = 3600 * 8
YAPPY_TASK_TIMEOUT = 3600 * 9
WRAPPER_TASK_TIMEOUT = 3600 * 10


class AliceEvoYappyWrapper(sdk2.Task):
    """
        Handle Yappy Beta Creation
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = WRAPPER_TASK_TIMEOUT
        evo_in_target_vins_package = sdk2.parameters.Resource(
            "evo_in_target_vins_package",
            resource_type=VinsPackage
        )
        evo_in_vins_id = sdk2.parameters.String(
            "evo_in_vins_id",
            required=True,
            default="vins-xx-xx"
        )
        with sdk2.parameters.Output:
            evo_out_launched_yappy_task = sdk2.parameters.String(
                "evo_out_launched_yappy_task"
            )
            evo_out_yappy_beta = sdk2.parameters.String(
                "evo_out_yappy_beta"
            )
            evo_out_yappy_url = sdk2.parameters.String(
                "evo_out_yappy_url"
            )
            evo_out_yappy_page = sdk2.parameters.String(
                "evo_out_yappy_page"
            )

    def on_execute(self):
        with self.memoize_stage.prepare_and_run_yappy_task:
            subtasks = []
            task = sdk2.Task["CREATE_YAPPY_BETA"](
                self,
                template_name="alice-megamind-rc",
                beta_suffix=self.Parameters.evo_in_vins_id,
                resources={
                    "package": self.Parameters.evo_in_target_vins_package.id,
                },
                beta_consistency_timeout=BETA_CONSISTENCY_TIMEOUT,
                description=self.Parameters.evo_in_target_vins_package,
                kill_timeout=YAPPY_TASK_TIMEOUT
            )
            logger.info("[INFO] After task creation")
            task.enqueue()
            logger.info("[INFO] Task enqueue")
            logger.info("[INFO] Task id: " + str(task.id))
            self.Parameters.evo_out_launched_yappy_task = \
                SandboxToolbox.get_sandbox_task_url(str(task.id))
            subtasks.append(task)
            raise sdk2.WaitTask(subtasks,
                                WAIT_STATUSES,
                                wait_all=True)

        with self.memoize_stage.after_yappy_task:
            subtasks = list(self.find())
            if subtasks and \
               any(task.status != ctt.Status.SUCCESS for task in subtasks):
                raise common_errors.TaskFailure("Yappy task failed!")

            self.Parameters.evo_out_yappy_beta = \
                YappyToolbox.get_beta_name_by_vins_id(self.Parameters.evo_in_vins_id)

            self.Parameters.evo_out_yappy_url = \
                YappyToolbox.get_megamind_speechkit_app_pa_url_by_vins_id(self.Parameters.evo_in_vins_id)

            self.Parameters.evo_out_yappy_page = \
                YappyToolbox.add_yappy_page_prefix(self.Parameters.evo_out_yappy_beta)
