# -*- coding: utf-8 -*-

import logging

from sandbox import sdk2
import sandbox.projects.release_machine.core.task_env as task_env

from sandbox.projects.alice_evo.common.const import AccessTokens
from sandbox.projects.alice_evo.common.const import EvoConstants
from sandbox.projects.alice_evo.common.release_machine import ReleaseMachineToolbox
from sandbox.projects.alice_evo.common.yappy import YappyToolbox

logger = logging.getLogger(__name__)


class BegemotMegamindEvoContext(sdk2.Task):
    '''
        Context getter for Begemot Megamind in release machine
    '''

    class Requirements(sdk2.Task.Requirements):
        environments = [task_env.TaskRequirements.startrek_client]
        client_tags = \
            task_env.TaskTags.startrek_client

        class Caches(sdk2.Requirements.Caches):
            pass

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600  # 10 min

        beta_prefix = sdk2.parameters.String(
            'beta_prefix',
            default='begemot-megamind',
        )
        beta_type = sdk2.parameters.String(
            'beta_type',
            default='alice-begemot-megamind',
        )
        backend_name = sdk2.parameters.String(
            'backend_name',
            default='ALICE__BEGEMOT_WORKER_MEGAMIND',
        )
        beta_url_prefix = sdk2.parameters.String(
            'beta_url_prefix',
            default='http://vins.hamster.alice.yandex.net/speechkit/app/pa/',
        )
        fallback_release_ticket = sdk2.parameters.String(
            'fallback_release_ticket',
            # default ticket for begemot-megamind ci
            default='MEGAMIND-3498',
        )
        evo_in_branch_num = sdk2.parameters.String(
            'evo_in_branch_num',
            required=True,
        )
        evo_in_tag_num = sdk2.parameters.String(
            'evo_in_tag_num',
            required=True,
        )
        evo_in_component_name = sdk2.parameters.String(
            'evo_in_component_name',
            default=EvoConstants.BEGEMOT_MEGAMIND_RM_NAME,
        )
        with sdk2.parameters.Output:
            evo_out_release_ticket = sdk2.parameters.String(
                'evo_out_release_ticket',
            )
            evo_out_beta_name = sdk2.parameters.String(
                'evo_out_beta_name',
            )
            evo_out_begemot_megamind_url = sdk2.parameters.String(
                'evo_out_begemot_megamind_url',
            )
            evo_out_yappy_page = sdk2.parameters.String(
                'evo_out_yappy_page',
            )
            evo_out_beta_slots = sdk2.parameters.String(
                'evo_out_beta_slots'
            )
            last_released_branch_num = sdk2.parameters.String(
                'last_released_branch_num'
            )
            last_released_tag_num = sdk2.parameters.String(
                'last_released_tag_num'
            )
            last_released_beta_name = sdk2.parameters.String(
                'last_released_beta_name'
            )

    def on_execute(self):
        logger.info('[INFO] Start task')

        # get last released state from begemot megamind executable resource
        begemot_executable = sdk2.Resource["BEGEMOT_MEGAMIND_EXECUTABLE"].find(state='READY', attrs={'released': 'stable'}).first()
        self.Parameters.last_released_branch_num, self.Parameters.last_released_tag_num = \
            ReleaseMachineToolbox.get_last_release_info(self.Parameters.evo_in_component_name, begemot_executable)
        self.Parameters.last_released_beta_name = YappyToolbox.get_beta_name(
            self.Parameters.beta_prefix,
            self.Parameters.last_released_branch_num,
            self.Parameters.last_released_tag_num
        )

        try:
            self.Parameters.evo_out_release_ticket = \
                ReleaseMachineToolbox.get_release_ticket_for_branch(
                    self.Parameters.evo_in_component_name,
                    self.Parameters.evo_in_branch_num
                )
        except:
            self.Parameters.evo_out_release_ticket = self.Parameters.fallback_release_ticket

        #  copy values to the context for rm config compatibility
        self.Context.evo_ctx_branch_num = self.Parameters.evo_in_branch_num
        self.Context.evo_ctx_tag_num = self.Parameters.evo_in_tag_num
        self.Context.evo_ctx_release_ticket = self.Parameters.evo_out_release_ticket
        self.Context.evo_ctx_component_name = self.Parameters.evo_in_component_name

        self.Parameters.evo_out_beta_name = YappyToolbox.get_beta_name(
            self.Parameters.beta_prefix,
            self.Parameters.evo_in_branch_num,
            self.Parameters.evo_in_tag_num
        )

        yappy_token = sdk2.Vault.data(AccessTokens.YAPPY_TOKEN_OWNER, AccessTokens.YAPPY_TOKEN_NAME)

        beta_slots = YappyToolbox.get_yappy_slots(
            self.Parameters.evo_out_beta_name,
            yappy_token,
            self.Parameters.beta_type,
        )
        logger.info('Slots for beta {} : {}'.format(self.Parameters.evo_out_beta_name, beta_slots))
        assert len(beta_slots) > 0
        self.Context.evo_ctx_beta_slots = beta_slots
        self.Parameters.evo_out_beta_slots = beta_slots
        # Yappy URL, e.g.:
        # http://vins.hamster.alice.yandex.net/speechkit/app/pa/?srcrwr=ALICE__BEGEMOT_WORKER_MEGAMIND:yappy-host-1.alice.yandex.net:81
        def make_begemot_srcrwr(slot):
            logger.info('Parse slot {}'.format(slot))
            host, _ = slot.split(":")
            port = 81
            return "srcrwr={}:{}:{}:{}".format(self.Parameters.backend_name, host, port, 10000)
        srcrwrs = "&".join(map(make_begemot_srcrwr, beta_slots))
        self.Parameters.evo_out_begemot_megamind_url = self.Parameters.beta_url_prefix + "?" + srcrwrs

        self.Parameters.evo_out_yappy_page = \
            YappyToolbox.get_yappy_page(
                self.Parameters.beta_prefix,
                self.Parameters.evo_in_branch_num,
                self.Parameters.evo_in_tag_num,
            )

        logger.info('[INFO] Finish task')
