# -*- coding: utf-8 -*-
import logging
import json
try:
    from urllib.parse import quote
except ImportError:
    from urllib import quote

import requests
from sandbox import sdk2
from sandbox.projects.common import binary_task


ST_COMMENT_TITLE_TEMPLATE = '===={title} Fails {version}===='
NO_AUTHOR = 'NO AUTHOR'
logger = logging.getLogger(__name__)


def _create_test_message(tests_info):
    tests = list(map(lambda test: 'Test: {name}\nSetrace: {setrace}\nLog: {log}\n{snippet}'.format(
        name=test.Name,
        setrace=test.SetraceLink,
        log=test.LogLink,
        snippet='!!{}!!\n'.format(test.Snippet) if test.Snippet else '',
    ), tests_info[1]))
    return '{author_name}\n<{{{test_count} failed tests\n{tests}}}>\n'.format(
        author_name=', '.join(map(lambda author: 'кто:{author}'.format(author=author), json.loads(tests_info[0]))),
        test_count=len(tests),
        tests='\n'.join(tests),
    )


def _create_startrek_message(task_id, title, text, tests=None):
    task_link = 'https://sandbox.yandex-team.ru/task/{}'.format(task_id)
    return '{title}\n\n<{{{task_link}\n{error_report}\n{release_error_report}}}>\n\n{text}\n\n{tests}'.format(
        title=title,
        task_link='Task\nTask: {}'.format(task_link),
        error_report='Error report: {}/error_report'.format(task_link),
        release_error_report='Release error report: {}/release_error_report'.format(task_link),
        text=text,
        tests='\n'.join(map(_create_test_message, tests.items())) if tests else '',
    )


def _update_to_authors_dict(authors_and_tests):
    updated_authors = {}
    default_authors = json.dumps([NO_AUTHOR])
    for authors, tests in authors_and_tests:
        authors_hash = default_authors
        if authors:
            authors.sort()
            authors_hash = json.dumps(authors)
        if authors_hash not in updated_authors:
            updated_authors[authors_hash] = []
        updated_authors[authors_hash].extend(tests)
    return updated_authors


def _get_logins_from_dict(data):
    logins = set()
    for item in data:
        login = item.get('person', {}).get('login')
        if login:
            logins.add(login)
    return logins


class CallEvoTestOwners(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Binary task for calling owners of EVO tests or duties.
    """

    class Parameters(sdk2.Task.Parameters):
        task_id = sdk2.parameters.Integer('Task id', required=True)
        release_ticket = sdk2.parameters.String('VINS release ticket', required=True)
        test_name = sdk2.parameters.String('Test package name (use in ST comment title)', required=True)
        only_release = sdk2.parameters.Bool('Only release fails', default=False)
        fail_threshold = sdk2.parameters.Integer('Min failed release tests for achtung message', default=100)
        release_version = sdk2.parameters.String('Release version', default='')

        with sdk2.parameters.Group('ARCANUM_TOKEN from Sandbox Vault') as yav_block:
            arcanum_token_name = sdk2.parameters.String('Name', required=True, default='robot-bassist_arcanum_token')
            arcanum_token_owner = sdk2.parameters.String('Owner', required=True, default='robot-bassist')

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        from alice.tests.evo_parser.lib import EvoFailsParser
        from alice.tests.evo_parser.lib.evo_parser_data_pb2 import TEvoParserRequest
        from google.protobuf.json_format import MessageToDict

        evo_fails_request = TEvoParserRequest(
            ArcanumToken=sdk2.Vault.data(self.Parameters.arcanum_token_owner, self.Parameters.arcanum_token_name),
            SandboxTaskId=self.Parameters.task_id,
            OnlyRelease=self.Parameters.only_release,
        )
        logger.info('EvoFailsRequest: {}'.format(json.dumps(MessageToDict(evo_fails_request))))

        evo_fails_response = EvoFailsParser(evo_fails_request).parse()
        logger.info('EvoFailsResponse: {}'.format(json.dumps(MessageToDict(evo_fails_response))))

        authors_and_tests = self._get_summoners_with_tests(evo_fails_response)
        authors_and_tests = _update_to_authors_dict(authors_and_tests)
        self._send_startrek_notification(authors_and_tests)

    def _get_people_from_abc(self, abc_name):
        oauth_token = 'OAuth ' + sdk2.Vault.data(self.Parameters.arcanum_token_owner, self.Parameters.arcanum_token_name)

        response = requests.get(
            url='https://abc-back.yandex-team.ru/api/v4/duty/on_duty/?service__slug={}'.format(abc_name),
            headers={'Authorization': oauth_token},
        )
        logging.info('OnDuty {}: {}'.format(abc_name, response.text))
        member_logins = _get_logins_from_dict(response.json())

        if not member_logins:
            response = requests.get(
                url='https://abc-back.yandex-team.ru/api/v4/services/members/?service__slug={}'.format(abc_name),
                headers={'Authorization': oauth_token},
            )
            logging.info('Abc members {}: {}'.format(abc_name, response.text))
            member_logins = _get_logins_from_dict(response.json().get('results', []))

        login = quote(','.join(member_logins))
        response = requests.get(
            url='https://staff-api.yandex-team.ru/v3/persons?official.is_dismissed=true&_fields=login&login={}'.format(login),
            headers={'Authorization': oauth_token},
        )
        dismissed_logins = set()
        for elem in response.json().get('result', []):
            login = elem.get('login')
            if login:
                dismissed_logins.add(login)

        return list(member_logins - dismissed_logins)

    def _get_suitable_authors(self, authors):
        suitable_authors = set()
        for author in authors:
            if author.StaffLogin.startswith('abc:'):
                abc_authors = self._get_people_from_abc(author.StaffLogin[len('abc:'):])
                suitable_authors.update(abc_authors[:1])
            elif not author.AbsenseType and author.StaffLogin.strip():
                suitable_authors.add(author.StaffLogin.strip())
        logging.info('Authors: {}'.format(suitable_authors))
        return list(suitable_authors)

    def _get_summoners_with_tests(self, evo_fails_response):
        return [(self._get_suitable_authors(pack.Authors), pack.Tests) for pack in evo_fails_response.FailedTestsPacks]

    def _send_startrek_notification(self, authors_and_tests):
        from startrek_client import Startrek
        st_client = Startrek(token=sdk2.Vault.data('robot_bassist_startrek_oauth_token'), useragent='robot-bassist')
        issue = st_client.issues[self.Parameters.release_ticket]

        summonees = [issue.assignee]
        task_id = self.Parameters.task_id
        test_count = sum(len(tests) for tests in authors_and_tests.values())
        title = ST_COMMENT_TITLE_TEMPLATE.format(title=self.Parameters.test_name, version=self.Parameters.release_version)
        if not test_count:
            comment_text = _create_startrek_message(task_id, title, text='!!(green){} are OK!!'.format(self.Parameters.test_name))
        else:
            comment_text = _create_startrek_message(task_id, title, text='!!{} failed tests detected!!'.format(test_count), tests=authors_and_tests)
            if test_count < self.Parameters.fail_threshold:
                authors = set(summonees)
                for author in authors_and_tests.keys():
                    authors.update(filter(lambda x: not x.startswith(('g:', NO_AUTHOR)), json.loads(author)))
                summonees = list(authors)

        issue.comments.create(text=comment_text, summonees=summonees)
