from datetime import datetime
import json
import logging
import random
import requests
import string
import time

from urllib import quote

from requests.packages.urllib3.exceptions import InsecureRequestWarning

from sandbox.projects.alice_evo.common.const import (AccessTokens,
                                                     EvoConstants,
                                                     NirvanaConstants)
from sandbox.projects.alice_evo.common.misc import MiscToolbox
from sandbox.projects.alice_evo.common.startrek import StartrekToolbox
from sandbox.projects.alice_evo.common.vault import VaultToolbox
from sandbox.common import errors as common_errors

logger = logging.getLogger(__name__)
TARGET_NIRVANA_WORKFLOW = "13e78538-c570-4d82-aac1-5c7fa5e2ad71"

TOKENS_ERROR = 'Failed to get tokens from vault!'

APPROVED_FOR_PRIEMKINS_RELEASES = [EvoConstants.ALICE_GRAPHS_RM_NAME,
                                   EvoConstants.BEGEMOT_MEGAMIND_RM_NAME,
                                   EvoConstants.HOLLYWOOD_COMMON_RM_NAME,
                                   EvoConstants.HOLLYWOOD_GENERAL_CONVERSATION_RM_NAME,
                                   EvoConstants.HOLLYWOOD_GOODWIN_RM_NAME,
                                   EvoConstants.VINS_RM_NAME,
                                   EvoConstants.UNIPROXY_RM_NAME,
                                   EvoConstants.ASR_RM_NAME,
                                   EvoConstants.VIDEOBASS_RM_NAME,
                                   EvoConstants.VOICE_GRAPHS_RM_NAME]


class NirvanaToolbox(object):
    @staticmethod
    def get_token_name(token_type, component_name):
        """
        Get proper token name by token type
        e.g.: "robot-bassist_arcanum_token" for "arcanum_token"

        """

        if token_type == "yql_token":
            if component_name in APPROVED_FOR_PRIEMKINS_RELEASES:
                return NirvanaConstants.TOKEN_YQL_PRIEMKINS
            else:
                return NirvanaConstants.TOKEN_YQL_BASSIST

        return ""

    @staticmethod
    def get_nirvana_token(component_name):
        if component_name in APPROVED_FOR_PRIEMKINS_RELEASES:
            return VaultToolbox.get_token_from_env_or_vault(
                AccessTokens.NIRVANA_PRIEMKINS_TOKEN_OWNER,
                AccessTokens.NIRVANA_PRIEMKINS_TOKEN_NAME,
                AccessTokens.NIRVANA_PRIEMKINS_TOKEN_ENV
            )

        return VaultToolbox.get_token_from_env_or_vault(
            AccessTokens.NIRVANA_TOKEN_OWNER,
            AccessTokens.NIRVANA_TOKEN_NAME,
            AccessTokens.NIRVANA_TOKEN_ENV
        )

    @staticmethod
    def _get_nirvana_quota(component_name):
        if component_name in [EvoConstants.VINS_RM_NAME]:
            return NirvanaConstants.MEGAMIND_NEW_QUOTA_PROJECT_ID

        return NirvanaConstants.NEW_QUOTA_PROJECT_ID

    @staticmethod
    def _get_yt_directory_suffix(
        build_id=None,
    ):
        random_suffix = \
            ''.join(random.choice(string.ascii_lowercase + string.ascii_uppercase + string.digits) for _ in range(
                NirvanaConstants.RANDOM_POSTFIX_LENGTH))

        yt_directory_suffix = \
            "_" + str(build_id) + "_" + \
            random_suffix

        return yt_directory_suffix

    @staticmethod
    def _get_nirvana_headers(robot_nirvana_token):
        return {'Authorization': 'OAuth ' + robot_nirvana_token, 'Content-type': 'application/json'}

    @staticmethod
    def _clone_nirvana_instance(
        robot_nirvana_token=None,
        workflow_instance_id=None,
        component_name=None
    ):
        logger.info("clone_nirvana_instance")

        if not robot_nirvana_token:
            logger.info("lack of important fields")
            return None

        workflow_id = NirvanaConstants.ETALON_UE2E_WORKFLOW
        target_workflow_id = NirvanaConstants.TARGET_UE2E_WORKFLOW

        headers = NirvanaToolbox._get_nirvana_headers(robot_nirvana_token)
        data = {"jsonrpc": "2.0",
                "params":
                    {  # Before ALICEINFRA-372 used "workflowInstanceId": workflow_instance_id
                        "workflowId": workflow_id,
                        "targetWorkflowId": target_workflow_id,
                        "newQuotaProjectId": NirvanaToolbox._get_nirvana_quota(component_name)
                    }
                }

        if workflow_instance_id:
            data["params"]["workflowInstanceId"] = workflow_instance_id

        requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
        r = requests.get(NirvanaConstants.NIRVANA_API + 'cloneWorkflowInstance',
                         headers=headers,
                         json=json.dumps(data),
                         params=data['params'],
                         verify=False)

        logger.info(str(r.status_code))
        if r.status_code == 200:
            result = r.json()
            logger.info(str(result))
            run_id = result.get('result', 0)
            if run_id:
                return str(run_id)  # workflowInstanceId of the new instance

        return None

    @staticmethod
    def _start_nirvana_instance(robot_nirvana_token=None, target_instance=None):
        logger.info("start_nirvana_instance")
        if not (robot_nirvana_token and target_instance):
            logger.info("lack of important fields")
            return None

        headers = NirvanaToolbox._get_nirvana_headers(robot_nirvana_token)

        data = {"jsonrpc": "2.0",
                "params": {"workflowInstanceId": target_instance,
                           "workflowId": TARGET_NIRVANA_WORKFLOW}}

        requests.packages.urllib3.disable_warnings(InsecureRequestWarning)
        r = requests.get(NirvanaConstants.NIRVANA_API + 'startWorkflow',
                         headers=headers,
                         json=json.dumps(data),
                         params=data['params'],
                         verify=False)
        logger.info(str(r.status_code))

        if r.status_code == 200:
            result = r.json()
            logger.info(str(result))
            run_id = result.get('result', 0)
            if run_id:
                return str(run_id)

        return None

    @staticmethod
    def _global_update_nirvana_comment(robot_nirvana_token=None, target_instance=None, target_comment=None):
        logger.info("global_update_nirvana_instance")
        if not (robot_nirvana_token and target_instance and target_comment):
            logger.info("lack of important fields")
            return None

        api_operation = "addCommentToWorkflowInstance"
        data = {"jsonrpc": "2.0",
                "method": api_operation,
                "id": TARGET_NIRVANA_WORKFLOW,
                "params": {"workflowInstanceId": target_instance,
                           "comment": target_comment}}

        result = NirvanaToolbox._post_json_to_nirvana_api(api_operation, data, robot_nirvana_token)

        if result:
            run_id = result.get('result', -1)

            if run_id != -1:
                return str(run_id)

        return None

    @staticmethod
    def _post_json_to_nirvana_api(api_operation, data, robot_nirvana_token):
        headers = NirvanaToolbox._get_nirvana_headers(robot_nirvana_token)
        requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

        r = requests.post(NirvanaConstants.NIRVANA_API + api_operation,
                          headers=headers,
                          json=data,
                          verify=False)

        logger.info(str(r.status_code))
        if r.status_code == 200:
            result = r.json()
            logger.info(str(result))
            return result

        return None

    @staticmethod
    def _encode_experiments(experiments):
        if not isinstance(experiments, list):
            return []
        res = []
        for exp in experiments:
            if exp:
                res.append(quote(exp) + "=1")
        return res

    @staticmethod
    def _global_update_nirvana_instance(robot_nirvana_token=None,
                                        target_instance=None,
                                        megamind_url=None,
                                        abc_id=None,
                                        test_experiments=None,
                                        component_name=None,
                                        yt_directory_suffix=None,
                                        is_trunk=None,
                                        uniproxy_url=None,
                                        app_type=None,
                                        release_ticket=None,
                                        fast_run=None,
                                        yt_tables_ttl=None,
                                        web_search_request_mode=None,
                                        baskets=None):
        logger.info("global_update_nirvana_instance")
        if not (robot_nirvana_token and target_instance and megamind_url and abc_id and component_name and yt_directory_suffix and uniproxy_url):
            logger.info("lack of important fields:")
            logger.info([x is not None for x in (robot_nirvana_token, target_instance, megamind_url, abc_id, component_name, yt_directory_suffix, uniproxy_url)])
            return None

        # Guide for values
        # https://wiki.yandex-team.ru/voicetechnology/dev/analytics/ue2e/priemka/
        if not is_trunk:
            var_priority = 95
            var_scraper_over_yt_pool = "alice-acceptance"
            var_scraper_token = "robot-alicesoyaccept-soy-token"
        else:
            var_priority = 50
            var_scraper_over_yt_pool = "alice"
            var_scraper_token = "robot-voice-qa-soy-token"

        if web_search_request_mode is None:
            web_search_request_mode = "0"

        api_operation = "setGlobalParameters"
        data = {"jsonrpc": "2.0",
                "method": api_operation,
                "id": TARGET_NIRVANA_WORKFLOW,
                "params": {"workflowInstanceId": target_instance,
                           "params": [
                               {"parameter": "prod_url",
                                "value": EvoConstants.MEGAMIND_HAMSTER_URL},

                               {"parameter": "test_url",
                                "value": megamind_url},
                               # e.g.: "http://megamind-rc.alice.yandex.net/speechkit/app/pa/"

                               {"parameter": "priority",
                                "value": var_priority},

                               {"parameter": "abc_id",
                                "value": abc_id},

                               {"parameter": "yql_token",
                                "value": NirvanaToolbox.get_token_name(
                                    "yql_token",
                                    component_name)
                                },

                               {"parameter": "mr_account",
                                "value": "alice"},

                               {"parameter": "ScraperOverYtPool",
                                "value": var_scraper_over_yt_pool},

                               {"parameter": "Scraper-token",
                                "value": var_scraper_token},

                               {"parameter": "cache_sync",
                                "value": int(time.time())},

                               {"parameter": "test_uniproxy_url",
                                "value": uniproxy_url},

                               {"parameter": "app_type",
                                "value": app_type},

                               {"parameter": "startrek_ticket",
                                "value": release_ticket},

                               {"parameter": "fast_run",
                                "value": fast_run},

                               {"parameter": "web_search_request_mode",
                                 "value": str(web_search_request_mode)},

                               {"parameter": "baskets",
                                "value": baskets},
                           ]}}
        if yt_tables_ttl is not None:
            data["params"]["params"] += [
                {"parameter": "mr-output-ttl",
                 "value": yt_tables_ttl},

                {"parameter": "final-results-ttl",
                 "value": yt_tables_ttl},
            ]

        if test_experiments is not None and test_experiments != []:
            data["params"]["params"].append({"parameter": "test_experiments",
                                             "value": NirvanaToolbox._encode_experiments(test_experiments)})

        logger.info(json.dumps(data, indent=4))

        result = NirvanaToolbox._post_json_to_nirvana_api(api_operation, data, robot_nirvana_token)

        if result:
            run_id = result.get('result', -1)

            if run_id != -1:
                return str(run_id)

        return None

    @staticmethod
    def handle_nirvana_launch(component_name=None,
                              megamind_url=None,
                              abc_id=None,
                              branch_num=None,
                              tag_num=None,
                              test_experiments=None,
                              force_start=None,
                              st_append_mode=None,
                              release_ticket=None,
                              is_trunk=None,
                              trunk_revision=None,
                              varelease_mode=None,
                              target_instance=None,
                              uniproxy_url=None,
                              app_type=None,
                              fast_run=None,
                              yt_tables_ttl=None,
                              web_search_request_mode=None,
                              baskets=None,
                              first_tag_num="1"):

        if varelease_mode:
            lower_st_task = release_ticket.lower()
            parts = lower_st_task.split("/")
            build_id = parts[-1]
        elif is_trunk:
            build_id = MiscToolbox.get_trunk_id(trunk_revision, component_name)
        else:
            build_id = MiscToolbox.get_component_release_id(
                component_name,
                branch_num,
                tag_num
            )

        nirvana_token = NirvanaToolbox.get_nirvana_token(
            component_name
        )

        if not nirvana_token:
            logger.error(TOKENS_ERROR)
            raise common_errors.TaskFailure(TOKENS_ERROR)

        rc_copy = NirvanaToolbox._clone_nirvana_instance(nirvana_token, target_instance, component_name)
        logger.info(rc_copy)
        nirvana_url_combo = \
            'https://nirvana.yandex-team.ru/flow/'
        nirvana_url_combo = nirvana_url_combo + \
            NirvanaConstants.TARGET_UE2E_WORKFLOW + '/'
        nirvana_url_combo = nirvana_url_combo + str(rc_copy)
        result_nirvana_instance_url = nirvana_url_combo

        robot_nirvana_token = nirvana_token

        yt_directory_suffix = NirvanaToolbox._get_yt_directory_suffix(
            build_id
        )
        rc_upd = NirvanaToolbox._global_update_nirvana_instance(
            robot_nirvana_token,
            rc_copy,
            megamind_url,
            abc_id,
            test_experiments,
            component_name,
            yt_directory_suffix,
            is_trunk,
            uniproxy_url,
            app_type,
            release_ticket,
            fast_run,
            yt_tables_ttl,
            web_search_request_mode,
            baskets,
        )

        rc_upd2 = \
            NirvanaToolbox._global_update_nirvana_comment(
                robot_nirvana_token,
                rc_copy,
                build_id
            )

        need_start = False
        if force_start or tag_num == first_tag_num:
            need_start = True

        if need_start:
            NirvanaToolbox._start_nirvana_instance(
                robot_nirvana_token,
                rc_copy)

        comment = NirvanaToolbox._generate_comment(build_id, need_start, nirvana_url_combo)

        if st_append_mode:
            from startrek_client import Startrek
            st_client = Startrek(token=StartrekToolbox.get_startrek_token(),
                                 useragent=AccessTokens.ST_USERAGENT)

            StartrekToolbox.append_startrek_comment(
                st_client,
                comment,
                release_ticket,
                NirvanaConstants.ST_COMMENT_TITLE,
                False
            )
        else:
            StartrekToolbox.post_single_comment(
                comment,
                release_ticket
            )

        logger.info(rc_upd)
        logger.info(rc_upd2)

        return result_nirvana_instance_url

    @staticmethod
    def _generate_comment(build_id,
                          need_start,
                          nirvana_url_combo):
        """
        Generate comment
        with cut, summary and link to report

        """
        comment = "[" + datetime.today().strftime("%Y-%m-%d %H:%M:%S") + "] " + \
            "UE2E for " + build_id + " "

        if need_start:
            comment += "has been created and started"
        else:
            comment += "has been prepared (however not automatically started)"

        comment += ": " + nirvana_url_combo + "\n"

        return comment
