from datetime import datetime
import logging
import requests

from sandbox.projects.alice_evo.common.const import AccessTokens
from sandbox.projects.alice_evo.common.vault import VaultToolbox

logger = logging.getLogger(__name__)


class StartrekToolbox(object):
    @staticmethod
    def get_startrek_token():
        return VaultToolbox.get_token_from_env_or_vault(
            AccessTokens.ST_TOKEN_OWNER,
            AccessTokens.ST_TOKEN_NAME,
            AccessTokens.ST_TOKEN_ENV
        )

    @staticmethod
    def post_single_comment(comment_text, ticket_target):
        """
            Post single comment to target ticket

        """

        robot_token = StartrekToolbox.get_startrek_token()

        if robot_token:
            content = {'text': comment_text}
            custom_headers = {'Content-Type': 'application/json',
                              'Accept': 'application/json',
                              'Authorization': 'OAuth ' + robot_token}
            target = 'https://st-api.yandex-team.ru/v2/issues/' \
                     + str(ticket_target) \
                     + '/comments'
            logger.info(str(ticket_target))
            r = requests.post(target,
                              json=content,
                              headers=custom_headers)
            logger.info(str(r.status_code))

    @staticmethod
    def append_startrek_comment(st_client, st_text, st_ticket, st_comment_base, st_comment_need_trunk_modified):
        """
            Allows to create or append message

            :param st_client: instance of Startrek
            :param st_text: text to add
            :param st_ticket: target Startrek ticket
            :param st_comment_base: title which starts the comment
            :param st_comment_need_trunk_modified: use if year and month modifier is required
        """
        logger.info(str(st_ticket))
        if not st_ticket:
            return
        st_comment_title = StartrekToolbox.generate_title(st_comment_base, st_comment_need_trunk_modified)
        logger.info("Before finding issue")
        # How to get st_client?
        # First of all, need to create the environment
        #
        # import sandbox.sandboxsdk.environments as sdk_environments
        # class Requirements(sdk2.Task.Requirements):
        #     environments = [sdk_environments.PipEnvironment('startrek_client',
        #     version='2.3.0', custom_parameters=['--upgrade-strategy only-if-needed']), ]
        #
        # from startrek_client import Startrek
        # st_client = Startrek(token=sdk2.Vault.data('robot_bassist_startrek_oauth_token'),
        #                      useragent='robot-bassist')
        # where token is Startrek token and useragent is Staff name of the user
        issue = st_client.issues[st_ticket]
        logger.info("After finding issue")
        comment = None
        for issue_comment in issue.comments.get_all():
            if issue_comment.text.startswith(st_comment_title):
                comment = issue_comment
                break
        logger.info("After looking through comments")
        if comment:
            issue.comments[comment.id].update(text=comment.text + st_text)
        else:
            issue.comments.create(text='{}\n\n{}'.format(st_comment_title, st_text))

    @staticmethod
    def generate_title(title, need_trunk_modified):
        st_comment_title = "====" + title + "===="
        if need_trunk_modified:
            st_comment_title = \
                "====" + \
                title + \
                " " + \
                datetime.today().strftime("%Y-%m") + \
                "===="
        return st_comment_title

    @staticmethod
    def get_linked_assessors_ticket(st_client, st_ticket):
        issue = st_client.issues[st_ticket]
        for link in issue.links:
            if link.object.key.startswith("ALICEASSESSORS"):
                return str(link.object.key)
        return ""
